package ru.yandex.direct.internaltools.tools.communication;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.communication.repository.CommunicationEventVersionsRepository;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventsRepository;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.communication.model.CommunicationEventVersionStatusParameters;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.internaltools.tools.communication.util.SendPersonalCommunicationEventUtils.getEventIdValidator;
import static ru.yandex.direct.internaltools.tools.communication.util.SendPersonalCommunicationEventUtils.getEventIterationValidator;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Tool(
        name = "Форсированное изменение статуса",
        label = "communication_event_version_status",
        description = "Используется только для устранения неполадок.",
        consumes = CommunicationEventVersionStatusParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.EXECUTE)
@Category(InternalToolCategory.COMMUNICATION_PLATFORM)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
public class CommunicationEventVersionStatusForceUpdate implements BaseInternalTool<CommunicationEventVersionStatusParameters> {

    private final CommunicationEventsRepository eventsRepository;
    private final CommunicationEventVersionsRepository versionsRepository;

    @Autowired
    public CommunicationEventVersionStatusForceUpdate(
            CommunicationEventsRepository communicationEventsRepository,
            CommunicationEventVersionsRepository communicationEventVersionsRepository
    ) {
        eventsRepository = communicationEventsRepository;
        versionsRepository = communicationEventVersionsRepository;
    }

    @Override
    public ValidationResult<CommunicationEventVersionStatusParameters, Defect> validate(CommunicationEventVersionStatusParameters params) {
        ItemValidationBuilder<CommunicationEventVersionStatusParameters, Defect> builder = ItemValidationBuilder.of(params);
        builder.item(params.getEventId(), "event")
                .checkBy(getEventIdValidator(eventsRepository, emptyList()));
        builder.item(params.getIteration(), "iteration")
                .checkBy(getEventIterationValidator(versionsRepository, params.getEventId()));
        builder.item(params.getStatus(), "status")
                .check(notNull());
        return builder.getResult();
    }

    @Override
    public InternalToolResult process(CommunicationEventVersionStatusParameters params) {
        var cev = versionsRepository.getVersion(params.getEventId(), params.getIteration());
        var prevStatus = cev.getStatus();
        cev.setStatus(params.getStatus());
        String result = versionsRepository.update(cev, prevStatus) ? "OK" : "FAIL";
        return new InternalToolResult(result);
    }
}
