package ru.yandex.direct.internaltools.tools.communication;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventVersionsRepository;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventsRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.communication.model.CommunicationEventVersionsParameters;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.cypress.YPath;

import static ru.yandex.direct.communication.CommunicationHelper.parseUsers;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.unconditional;
import static ru.yandex.direct.validation.constraint.StringConstraints.notEmpty2;
import static ru.yandex.direct.validation.defect.CollectionDefects.inCollection;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;
import static ru.yandex.direct.ytwrapper.YtUtils.CONTENT_REVISION_ATTR;
import static ru.yandex.direct.ytwrapper.YtUtils.SCHEMA_ATTR;

@Tool(
        name = "Управление списками версий коммуникационных событий",
        label = "communication_event_versions",
        description = "Позволяет смотреть и управлять версиями событий.",
        consumes = CommunicationEventVersionsParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.EXECUTE)
@Category(InternalToolCategory.COMMUNICATION_PLATFORM)
@AccessGroup({InternalToolAccessRole.INTERNAL_USER})
@ParametersAreNonnullByDefault
public class CommunicationEventVersionsTool extends BaseCommunicationEventVersionsTool<CommunicationEventVersionsParameters> {

    @Autowired
    public CommunicationEventVersionsTool(
            CommunicationEventsRepository communicationEventsRepository,
            CommunicationEventVersionsRepository communicationEventVersionsRepository,
            YtProvider ytProvider,
            UserService userService
    ) {
        super(communicationEventsRepository, communicationEventVersionsRepository, ytProvider, userService);
    }

    @Override
    protected void fillVersionModelByInput(
            CommunicationEventVersion cev,
            CommunicationEventVersionsParameters params,
            boolean processInternal
    ) {
        super.fillVersionModelByInput(cev, params, processInternal);

        String users = params.getUsers();
        if (users != null) {
            cev.setUsers(users);
        }

        if (processInternal) {
            fillVersionModelByInputInternal(cev, params);
        }
    }

    @Override
    protected CommunicationEventVersionsParameters makeParamsByVersionModel(CommunicationEventVersion cev) {
        return new CommunicationEventVersionsParameters()
                .withEvent(cev.getEventId() + ": " + cev.getIter())
                .withIteration(cev.getIter())
                .withStartEventTime(cev.getStartTime())
                .withExpired(cev.getExpired())
                .withUsers(cev.getUsers())
                .withTitle(cev.getTitle())
                .withText(cev.getText())
                .withButtonText(cev.getButtonText())
                .withButtonHref(cev.getButtonHref())
                .withImageHref(cev.getImageHref());
    }

    protected ValidationResult<String, Defect> userValidator(String users) {
        ItemValidationBuilder<String, Defect> builder = ItemValidationBuilder.of(users);
        builder.check(notNull()).check(notEmpty2());
        if (StringUtils.isEmpty(users)) {
            return builder.getResult();
        }
        if (users.contains(":")) {
            String[] typeColumnClusterPath = parseUsers(users);
            builder.item(typeColumnClusterPath[0], "type")
                    .check(inSet(Set.of("uid", "login", "ClientID")));
            if (!YtCluster.getNames().contains(typeColumnClusterPath[2])) {
                builder.item(typeColumnClusterPath[2], "cluster")
                        .check(unconditional(inCollection()));
                return builder.getResult();
            }
            var cypress = ytProvider
                    .get(YtCluster.parse(typeColumnClusterPath[2]))
                    .cypress();
            var yTable = YPath.simple(typeColumnClusterPath[3]);
            if (!cypress.exists(yTable)) {
                builder.item(typeColumnClusterPath[3], "path")
                        .check(unconditional(invalidValue()));
                return builder.getResult();
            }
            Set<String> columns = new HashSet<>();
            cypress.get(yTable.attribute(SCHEMA_ATTR)).listNode().forEach(
                    c -> columns.add(c.mapNode().getOrThrow("name").stringValue())
            );
            builder.item(typeColumnClusterPath[1], "column")
                    .check(inSet(columns));
        } else {
            List<String> logins = StreamEx.of(users.split(","))
                    .map(String::trim)
                    .filter(s -> !StringUtils.isEmpty(s))
                    .toList();
            Set<String> existLogins = userService.massGetUserByLogin(logins)
                    .stream()
                    .map(User::getLogin)
                    .collect(Collectors.toSet());
            builder.list(logins, "logins")
                    .checkEach(inSet(existLogins), invalidValue());
        }

        return builder.getResult();
    }

    @Override
    protected ItemValidationBuilder<CommunicationEventVersionsParameters, Defect> createValidationBuilder(CommunicationEventVersionsParameters params) {
        var builder = super.createValidationBuilder(params);
        builder.item(params.getUsers(), "users").checkBy(this::userValidator);
        return builder;
    }

    private void fillVersionModelByInputInternal(
            CommunicationEventVersion cev,
            CommunicationEventVersionsParameters params) {
        String users = params.getUsers();
        if (users != null) {
            if (users.contains(":")) {
                String[] typeColumnClusterPath = parseUsers(users);
                YtCluster cluster = YtCluster.parse(typeColumnClusterPath[2]);
                YPath path = YPath.simple(typeColumnClusterPath[3]);
                var contentRevision = ytProvider
                        .get(cluster)
                        .cypress()
                        .get(path.attribute(CONTENT_REVISION_ATTR))
                        .longValue() + "";
                cev.setCluster(cluster.getName());
                cev.setUserTableHash(contentRevision);
            } else {
                cev.setCluster(null);
                cev.setUserTableHash(null);
            }
        }
    }


}
