package ru.yandex.direct.internaltools.tools.communication;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.clickhouse.SqlBuilder;
import ru.yandex.direct.core.entity.communication.model.CommunicationEvent;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventVersionsRepository;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventsRepository;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.dbutil.wrapper.SimpleDb;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolMassResult;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.communication.model.CommunicationStatisticParameters;
import ru.yandex.direct.internaltools.tools.communication.model.CommunicationStatisticResponse;
import ru.yandex.direct.logviewercore.domain.LogRecordInfo;
import ru.yandex.direct.logviewercore.domain.LogTablesInfoManager;
import ru.yandex.direct.logviewercore.domain.ppclog.LogRecord;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus.ABORTED;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus.ABORTING;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus.ACTIVATING;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus.ACTIVE;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus.ARCHIVED;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus.NEED_ABORT;
import static ru.yandex.direct.internaltools.tools.communication.util.SendPersonalCommunicationEventUtils.getEventIdValidator;

@Tool(
        name = "Сбор статистики по событиям",
        label = "communication_event_statistics",
        description = "Позволяет смотреть статистику событий",
        consumes = CommunicationStatisticParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.SHOW)
@Category(InternalToolCategory.COMMUNICATION_PLATFORM)
@AccessGroup({InternalToolAccessRole.INTERNAL_USER})
@ParametersAreNonnullByDefault
public class CommunicationStatisticTool implements BaseInternalTool<CommunicationStatisticParameters> {

    private static final Logger logger = LoggerFactory.getLogger(CommunicationStatisticTool.class);

    private static final String COLUMN_TIME = "interval_start_time";
    private static final String COLUMN_COUNT_DISTINCT = "user_count";
    private static final String COLUMN_BUTTON = "button";

    private static final String EXPR_MESSAGE_ID = "cast(splitByString(' ', message)[2] as UInt64)";
    private static final String EXPR_BUTTON = "splitByString(' ', message)[4]";
    private static final String EXPR_EVENT_ID = EXPR_MESSAGE_ID +
            " - 4294967296 * cast(floor(" + EXPR_MESSAGE_ID + " / 4294967296) as UInt64)";

    private final CommunicationEventsRepository eventsRepository;
    private final CommunicationEventVersionsRepository versionsRepository;
    private final DatabaseWrapperProvider dbProvider;

    @Autowired
    public CommunicationStatisticTool(
            CommunicationEventsRepository communicationEventsRepository,
            CommunicationEventVersionsRepository communicationEventVersionsRepository,
            DatabaseWrapperProvider databaseWrapperProvider
    ) {
        eventsRepository = communicationEventsRepository;
        versionsRepository = communicationEventVersionsRepository;
        dbProvider = databaseWrapperProvider;
    }

    @Override
    public ValidationResult<CommunicationStatisticParameters, Defect> validate(CommunicationStatisticParameters params) {
        ItemValidationBuilder<CommunicationStatisticParameters, Defect> builder = ItemValidationBuilder.of(params);
        builder.item(params.getEventId(), "event")
                .checkBy(getEventIdValidator(eventsRepository, Collections.emptyList()));
        return builder.getResult();
    }

    @Override
    public InternalToolResult process(CommunicationStatisticParameters params) {
        Map<LocalDateTime, CommunicationStatisticResponse> results = new HashMap<>();

        LogRecordInfo<? extends LogRecord> info = LogTablesInfoManager.getLogRecordInfo("messages");
        String timeExpression = getTimeFunction(params.getGroupByPeriod(), info.getLogTimeColumn());
        var startDate = params.getStartDate();
        if (startDate == null) {
            // По умолчанию берем ActivateTime события
            startDate = eventsRepository
                    .getCommunicationEventsByIds(List.of(params.getEventId()))
                    .stream()
                    .map(CommunicationEvent::getActivateTime)
                    .filter(Objects::nonNull)
                    .map(LocalDateTime::toLocalDate)
                    .findAny()
                    .orElse(LocalDate.now());
        }
        var finishDate = params.getFinishDate();
        if (finishDate == null) {
            var startedStatuses = List.of(ACTIVATING, ACTIVE, NEED_ABORT, ABORTING, ABORTED, ARCHIVED);
            // По умолчанию берем последнее время завершения итерации, которая была когда-либо активной
            finishDate = versionsRepository
                    .getVersionsByEvent(params.getEventId())
                    .stream()
                    .filter(v -> startedStatuses.contains(v.getStatus()))
                    .map(CommunicationEventVersion::getExpired)
                    .filter(Objects::nonNull)
                    .map(LocalDateTime::toLocalDate)
                    .max(LocalDate::compareTo)
                    .orElse(LocalDate.now());
        }

        SqlBuilder sumSql = buildSql(info.getTableName(), timeExpression,
                startDate, finishDate, params.getEventId(), false);

        dbProvider.get(SimpleDb.CLICKHOUSE_CLOUD)
                .query(sumSql.generateSql(false), sumSql.getBindings(), (rs, i) -> {
                    var time = rs.getTimestamp(COLUMN_TIME).toLocalDateTime();
                    var result = results.get(time);
                    if (result == null) {
                        result = new CommunicationStatisticResponse(time);
                        results.put(time, result);
                    }
                    result.clickUserCount += rs.getLong(COLUMN_COUNT_DISTINCT);
                    return null;
                });

        SqlBuilder buttonSql = buildSql(info.getTableName(), timeExpression,
                startDate, finishDate, params.getEventId(), true);
        dbProvider.get(SimpleDb.CLICKHOUSE_CLOUD)
                .query(buttonSql.generateSql(false), buttonSql.getBindings(), (rs, i) -> {
                    var time = rs.getTimestamp(COLUMN_TIME).toLocalDateTime();
                    var result = results.get(time);
                    if (result == null) {
                        result = new CommunicationStatisticResponse(time);
                        results.put(time, result);
                    }
                    if ("null".equals(rs.getString(COLUMN_BUTTON))) {
                        result.skipUserCount += rs.getLong(COLUMN_COUNT_DISTINCT);
                    } else {
                        result.redirectUserCount += rs.getLong(COLUMN_COUNT_DISTINCT);
                    }
                    return null;
                });

        return new InternalToolMassResult(EntryStream.of(results)
                .sorted(Comparator.comparing(Map.Entry::getKey))
                .values()
                .toList());
    }

    private String getTimeFunction(CommunicationStatisticParameters.GroupBy groupBy, String columnName) {
        switch (groupBy) {
            case NONE:
                return "toDateTime('0000-00-00 00:00:00')";
            case MINUTE:
                return String.format("toStartOfMinute(%s)", columnName);
            case MINUTE_5:
                return String.format("toStartOfFiveMinute(%s)", columnName);
            case HOUR:
                return String.format("toStartOfHour(%s)", columnName);
            case DAY:
                return String.format("toDateTime(toDate(%s))", columnName);
            case MONTH:
                return String.format("toDateTime(toStartOfMonth(%s))", columnName);
            case QUERTER:
                return String.format("toDateTime(toStartOfQuarter(%s))", columnName);
            case YEAR:
                return String.format("toDateTime(toStartOfYear(%s))", columnName);
            default:
                throw new IllegalArgumentException("Unsupported TimeGroupingType: " + groupBy);
        }
    }

    private SqlBuilder buildSql(
            String tableName, String timeExpression,
            LocalDate startDate, LocalDate finishDate,
            Long eventId, boolean groupByButton) {
        SqlBuilder sql = new SqlBuilder()
                .from(tableName)
                .selectExpression(String.format("count(distinct %s)", EXPR_MESSAGE_ID), COLUMN_COUNT_DISTINCT)
                .selectExpression(timeExpression, COLUMN_TIME);
        if (groupByButton) {
            sql.selectExpression(EXPR_BUTTON, COLUMN_BUTTON);
        }
        sql.where("log_date BETWEEN toDate(?) AND toDate(?)",
                DateTimeFormatter.ISO_LOCAL_DATE.format(startDate),
                DateTimeFormatter.ISO_LOCAL_DATE.format(finishDate))
                .where("service = 'direct.web'")
                .where("like(method, 'communication.messages.%.button')")
                .where("class_name = 'ru.yandex.direct.web.entity.communication.service.CommunicationWebService'")
                .where("log_level = 'INFO'")
                .where("like(message, 'messageID= %buttonId= %')")
                .groupByExpression(timeExpression)
                .groupByExpression(EXPR_EVENT_ID);
        if (groupByButton) {
            sql.groupByExpression(EXPR_BUTTON);
        }
        sql.having(EXPR_EVENT_ID + " = ?", eventId);
        StringBuilder log = new StringBuilder()
                .append("Query: ")
                .append(sql.generateSql(true))
                .append(" \nBindings: ");
        var bingings = sql.getBindings();
        for (int i = 0; i < bingings.length; i++ ) {
            log.append(i)
                    .append(": ")
                    .append(bingings[i])
                    .append(", ");
        }
        logger.info(log.toString());
        return sql;
    }
}
