package ru.yandex.direct.internaltools.tools.communication;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventVersionsRepository;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventsRepository;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.communication.model.PopupEventVersionsParameters;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtTable;

import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@Tool(
        name = "Управление списками версий popup событий",
        label = "popup_event_versions",
        description = "Позволяет смотреть и управлять версиями popup событий",
        consumes = PopupEventVersionsParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.EXECUTE)
@Category(InternalToolCategory.COMMUNICATION_PLATFORM)
@AccessGroup({InternalToolAccessRole.INTERNAL_USER})
@ParametersAreNonnullByDefault
public class PopupEventVersionsTool extends BaseCommunicationEventVersionsTool<PopupEventVersionsParameters> {
    @Autowired
    protected PopupEventVersionsTool(
            CommunicationEventsRepository eventsRepository,
            CommunicationEventVersionsRepository versionsRepository,
            YtProvider ytProvider,
            UserService userService
    ) {
        super(eventsRepository, versionsRepository, ytProvider, userService);
    }

    @Override
    protected void fillVersionModelByInput(
            CommunicationEventVersion cev,
            PopupEventVersionsParameters params,
            boolean processInternal) {
        super.fillVersionModelByInput(cev, params, processInternal);
        cev.setEventsTablePath(params.getTablePath());
        cev.setCluster(params.getCluster());

    }

    @Override
    protected ItemValidationBuilder<PopupEventVersionsParameters, Defect> createValidationBuilder(PopupEventVersionsParameters params) {
        var builder = super.createValidationBuilder(params);
        builder.checkBy(this::tablePathValidator);
        return builder;
    }

    @Override
    protected CommunicationEventVersionStatus statusOnStart() {
        return CommunicationEventVersionStatus.ACTIVE;
    }

    @Override
    protected CommunicationEventVersionStatus statusOnStop() {
        return CommunicationEventVersionStatus.NEW_;
    }

    @Override
    protected PopupEventVersionsParameters makeParamsByVersionModel(CommunicationEventVersion cev) {
        return new PopupEventVersionsParameters()
                .withEvent(cev.getEventId() + ": " + cev.getIter())
                .withIteration(cev.getIter())
                .withStartEventTime(cev.getStartTime())
                .withExpired(cev.getExpired())
                .withTitle(cev.getTitle())
                .withText(cev.getText())
                .withButtonText(cev.getButtonText())
                .withButtonHref(cev.getButtonHref())
                .withImageHref(cev.getImageHref())
                .withCluster(cev.getCluster())
                .withTablePath(cev.getEventsTablePath());
    }

    protected ValidationResult<PopupEventVersionsParameters, Defect> tablePathValidator(
            PopupEventVersionsParameters params
    ) {
        ItemValidationBuilder<PopupEventVersionsParameters, Defect> validator = ItemValidationBuilder.of(params);
        var cluster = params.getCluster() != null ? YtCluster.parse(params.getCluster()) : null;
        validator.item(cluster, "ytCluster")
                .check(CommonConstraints.notNull());
        if (validator.getResult().hasAnyErrors()) {
            return validator.getResult();
        }

        validator.item(params.getTablePath(), "tablePath")
                .check(CommonConstraints.notNull())
                .check(fromPredicate(tablePath -> ytProvider.getOperator(cluster).exists(new YtTable(tablePath)),
                        CommonDefects.objectNotFound()));
        return validator.getResult();
    }
}
