package ru.yandex.direct.internaltools.tools.communication;

import java.io.IOException;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventVersionsRepository;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventsRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolMassResult;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.communication.model.CommunicationEventVersionsAction;
import ru.yandex.direct.internaltools.tools.communication.model.RecommendationVersionsParameters;
import ru.yandex.direct.internaltools.tools.communication.model.RecommendationVersionsResponse;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.utils.FunctionalUtils;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.validation.defect.CommonDefects.inconsistentState;

@Tool(
        name = "Управление списками версий рекомендаций",
        label = "recommendation_versions",
        description = "Позволяет смотреть и управлять версиями рекомендаций.",
        consumes = RecommendationVersionsParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.EXECUTE)
@Category(InternalToolCategory.COMMUNICATION_PLATFORM)
@AccessGroup({InternalToolAccessRole.INTERNAL_USER})
@ParametersAreNonnullByDefault
public class RecommendationVersionsTool implements BaseInternalTool<RecommendationVersionsParameters> {

    private static final Set<String> APPROVERS = Set.of(
            "yndx-a-dubov",
            "yndx-bakin",
            "yndx-r-shakirova-developer"
    );

    private final CommunicationEventsRepository eventsRepository;
    private final CommunicationEventVersionsRepository versionsRepository;

    @Autowired
    public RecommendationVersionsTool(
            CommunicationEventsRepository communicationEventsRepository,
            CommunicationEventVersionsRepository communicationEventVersionsRepository
    ) {
        eventsRepository = communicationEventsRepository;
        versionsRepository = communicationEventVersionsRepository;
    }

    @Override
    public ValidationResult<RecommendationVersionsParameters, Defect> validate(RecommendationVersionsParameters params) {
        ItemValidationBuilder<RecommendationVersionsParameters, Defect> builder = ItemValidationBuilder.of(params);
        if (CommunicationEventVersionsAction.UPDATE.equals(params.getAction())) {
            builder.item(params.getOperator(), "rights")
                    .checkByFunction(this::validateApprover);
        }
        return builder.getResult();
    }

    private Defect validateApprover(User operator) {
        if (operator.getRole().equals(RbacRole.SUPER)) {
            return null;
        }
        if (APPROVERS.contains(operator.getLogin())) {
            return null;
        }
        return inconsistentState();
    }

    @Override
    public InternalToolResult process(RecommendationVersionsParameters params) {
        Long eventId = params.getEventId();
        Long versionId = params.getIteration();
        CommunicationEventVersion cev;
        switch (params.getAction()) {
            case CREATE:
                cev = parse(params.getConfig());
                cev.setEventId(eventId);
                cev.setIter(versionId);
                cev.setStatus(CommunicationEventVersionStatus.NEW_);
                cev.setCreationTimestamp(System.currentTimeMillis() / 1000);
                versionsRepository.create(cev);
                break;
            case UPDATE:
                cev = parse(params.getConfig());
                cev.setEventId(eventId);
                cev.setIter(versionId);
                versionsRepository.update(cev, Set.of(CommunicationEventVersionStatus.values()), false);
        }
        var results = versionId == null ?
                versionsRepository.getVersionsByEvent(eventId) :
                versionsRepository.getOptionalVersion(eventId, versionId)
                        .map(List::of).orElse(List.of());
        return new InternalToolMassResult(FunctionalUtils.mapList(results, RecommendationVersionsResponse::new));
    }

    private CommunicationEventVersion parse(String json) {
        try {
            return JsonUtils.getObjectMapper().readValue(json, CommunicationEventVersion.class);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
