package ru.yandex.direct.internaltools.tools.communication;

import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.communication.CommunicationChannelRepository;
import ru.yandex.direct.communication.CommunicationClient;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.communication.model.SendPersonalCommunicationEventParameters;
import ru.yandex.direct.internaltools.tools.communication.model.SendPersonalCommunicationEventResponse;
import ru.yandex.direct.internaltools.tools.communication.util.SendPersonalCommunicationEventUtils;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.utils.ThreadUtils;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;

@Tool(
        name = "Завести тестовое событие на свой аккаунт",
        label = "send_personal_event",
        description = "Позволяет создать и сразу же отправить тестовое событие в свой аккаунт.",
        consumes = SendPersonalCommunicationEventParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.CREATE)
@Category(InternalToolCategory.COMMUNICATION_PLATFORM)
@AccessGroup({InternalToolAccessRole.INTERNAL_USER})
@ParametersAreNonnullByDefault
public class SendPersonalCommunicationEventTool extends MassInternalTool<SendPersonalCommunicationEventParameters,
        SendPersonalCommunicationEventResponse> {

    private static final Logger logger = LoggerFactory.getLogger(SendPersonalCommunicationEventTool.class);
    private static final Long SLEEP_TIME = 3_000L;
    private static final Integer ATTEMPTS = 10;

    private final CommunicationClient communicationClient;
    private final CommunicationChannelRepository communicationChannelRepository;

    public SendPersonalCommunicationEventTool(
            TvmIntegration tvmIntegration,
            DirectConfig directConfig,
            CommunicationChannelRepository communicationChannelRepository) {
        this.communicationClient = new CommunicationClient(tvmIntegration, directConfig);
        this.communicationChannelRepository = communicationChannelRepository;
    }

    @Override
    public ValidationResult<SendPersonalCommunicationEventParameters, Defect> validate(SendPersonalCommunicationEventParameters params) {
        var vb = ItemValidationBuilder.of(params, Defect.class);
        if (params.getButtonHref() != null) {
            vb.item(params.getButtonHref(), "href")
                    .check(validHref());
        }
        if (params.getImageHref() != null) {
            vb.item(params.getImageHref(), "imageLink")
                    .check(validHref());
        }
        return vb.getResult();
    }

    @Override
    protected List<SendPersonalCommunicationEventResponse> getMassData(SendPersonalCommunicationEventParameters parameter) {
        Long operatorUid = parameter.getOperator().getUid();
        String messageJson = SendPersonalCommunicationEventUtils.createMessageText(parameter);
        var communicationEvent = SendPersonalCommunicationEventUtils
                .createSingleEvent(operatorUid, messageJson);
        long eventId = communicationEvent.getData().getId();

        try {
            communicationClient.send(communicationEvent);
        } catch (ExecutionException ex) {
            logger.error("Caught exception", ex);
            return singletonList(convertToIntTool(eventId, "FAIL: SENDING ERROR"));
        } catch (TimeoutException ex) {
            logger.error("Timeout", ex);
            return singletonList(convertToIntTool(eventId, "FAIL: TIMEOUT ERROR"));
        }

        for (int i = 0; i < ATTEMPTS; i++) {
            ThreadUtils.sleep(SLEEP_TIME);
            if (!filterList(communicationChannelRepository.getMessagesByUid(operatorUid, List.of(eventId)),
                    message -> message.getEventId() == eventId).isEmpty()) {
                return singletonList(convertToIntTool(eventId, "OK"));
            }
        }
        return singletonList(convertToIntTool(eventId, "TIMEOUT: NO TABLE ENTRY YET"));
    }

    private SendPersonalCommunicationEventResponse convertToIntTool(Long eventId, String isWritten) {
        return new SendPersonalCommunicationEventResponse()
                .withEventId(eventId)
                .withIsWritten(isWritten);
    }
}
