package ru.yandex.direct.internaltools.tools.communication.repository

import com.fasterxml.jackson.module.kotlin.readValue
import com.google.common.io.Resources
import java.nio.charset.StandardCharsets
import org.apache.commons.io.IOUtils
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Repository
import ru.yandex.direct.config.DirectConfig
import ru.yandex.direct.core.entity.communication.model.CommunicationEvent
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion
import ru.yandex.direct.internaltools.tools.communication.model.ParsedCommunicationConfiguration
import ru.yandex.direct.utils.JsonUtils
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.defect.CommonDefects.objectNotFound
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

class ConfigHeader(
    val folderName: String,
    val versionFiles: List<String>
)

@Repository
class CommunicationEventConfigurationRepository(config: DirectConfig) {
    private val eventFileName: String
    private val configPath: String
    private val headerFile: String
    private val logger = LoggerFactory.getLogger(CommunicationEventConfigurationRepository::class.java)

    init {
        configPath = config.getString("communication_config.path")
        eventFileName = config.getString("communication_config.event_file")
        headerFile = config.getString("communication_config.header")
        if (eventFileName == null || configPath == null || headerFile == null)
            throw IllegalArgumentException("The header file, the configuration path and the event file name are required")
    }

    fun getEventVersions(
        eventId: Long,
        iteration: Long
    ): ValidationResult<ParsedCommunicationConfiguration<CommunicationEventVersion>?, Defect<Void>> {
        val eventsResult = getEvents(eventId)
        if (eventsResult.hasAnyErrors() || eventsResult.value == null) {
            logger.error("Configuration for communication event, id: $eventId has not been found")
            return ValidationResult.failed(null, objectNotFound())
        }
        val parentDir = eventsResult.value!!.configurationPath
        val events = getConfigHeader()
            .filter { parentDir.endsWith("/${it.folderName}") }
            .flatMap { it.versionFiles }
            .map {
                val text = readFile("$parentDir/$it")
                val parsed = JsonUtils.getObjectMapper().readValue(text, CommunicationEventVersion::class.java)
                ParsedCommunicationConfiguration(it, parsed)
            }.filter { eventId == it.configuration.eventId && iteration == it.configuration.iter }

        if (events.size > 1) {
            logger.error("Warning! Multiple configuration files have been found for eventId: $eventId " +
                "and iteration number: $iteration")
            return ValidationResult.failed(null, CommonDefects.invalidValue())
        }
        return ValidationResult.success(events.first())
    }

    fun getEvents(
        eventId: Long
    ): ValidationResult<ParsedCommunicationConfiguration<CommunicationEvent>?, Defect<Void>> {
        val configurationDirectories = getConfigHeader().map { configPath + it.folderName }
        if (configurationDirectories.isEmpty()) {
            logger.error("Communication configurations will not be processed due to $configPath has not been found")
            return ValidationResult.failed(null, objectNotFound())
        }

        val events = configurationDirectories
            .map {
                val text = readFile("$it/$eventFileName")
                val result = JsonUtils.getObjectMapper().readValue(text, CommunicationEvent::class.java)
                ParsedCommunicationConfiguration(it, result)
            }
            .filter { eventId == it.configuration.eventId }

        if (events.size > 1) {
            logger.error("Multiple configuration files have been found for eventId: $eventId")
            return ValidationResult.failed(null, CommonDefects.invalidValue())
        }

        return ValidationResult.success(events.firstOrNull())
    }

    private fun readFile(path: String): String {
        val inStream = Resources.getResource(path).openStream()
        val contents: String = IOUtils.toString(inStream, StandardCharsets.UTF_8)
        inStream.close()
        return contents
    }

    private fun getConfigHeader(): List<ConfigHeader> {
        val inStream = Resources.getResource(headerFile).openStream()
        val contents: String = IOUtils.toString(inStream, StandardCharsets.UTF_8)
        val objects: List<ConfigHeader> = JsonUtils.getObjectMapper().readValue(contents)
        inStream.close()
        return objects
    }
}
