package ru.yandex.direct.internaltools.tools.communication.service

import com.fasterxml.jackson.core.JsonProcessingException
import org.assertj.core.util.diff.DiffUtils
import org.slf4j.Logger
import ru.yandex.direct.internaltools.tools.communication.CommunicationConfigurationTool
import ru.yandex.direct.internaltools.tools.communication.model.CommunicationConfigurationAction
import ru.yandex.direct.internaltools.tools.communication.model.CommunicationConfigurationItem
import ru.yandex.direct.internaltools.tools.communication.model.CommunicationConfigurationParameters
import ru.yandex.direct.internaltools.tools.communication.model.ParsedCommunicationConfiguration
import ru.yandex.direct.rbac.RbacRole
import ru.yandex.direct.result.Result
import ru.yandex.direct.utils.JsonUtils
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

abstract class CommunicationConfigurationService<T> {
    private val executors = mapOf(
        CommunicationConfigurationAction.ARCHIVE to ::archiveEntry,
        CommunicationConfigurationAction.ACTIVATE to ::upsertEntry,
        CommunicationConfigurationAction.SHOW to ::showEntry,
        CommunicationConfigurationAction.COMPARE to ::compareEntry,
        CommunicationConfigurationAction.SHOW_CURRENT to ::showDbEntry,
    )

    abstract fun getType(): Class<T>
    protected abstract fun getEntryByParams(params: CommunicationConfigurationParameters): T
    abstract fun archiveEntry(config: ParsedCommunicationConfiguration<T>?, entry: T?): Result<Any>
    abstract fun upsertEntry(config: ParsedCommunicationConfiguration<T>, entry: T?): Result<Any>

    protected abstract fun getLogger(): Logger
    protected abstract fun getEntryConfigByParams(
        params: CommunicationConfigurationParameters): ParsedCommunicationConfiguration<T>?

    fun process(
        params: CommunicationConfigurationParameters
    ): Result<out Any> {
        val config = getEntryConfigByParams(params)
        val entry = getEntryByParams(params)
        val errors = validateAction(params, config, entry)
        if (errors.isNotEmpty()) {
            val validation: ValidationResult<Any, Defect<*>> =
                ValidationResult.failed(errors.joinToString(", "), CommonDefects.invalidValue());
            return Result.broken(validation)
        }

        return executors[params.action]!!.call(config, entry)
    }

    private fun validateAction(
        params: CommunicationConfigurationParameters,
        config: ParsedCommunicationConfiguration<T>?,
        entry: T
    ): List<String> {
        val errors = mutableListOf<String>()
        if (params.action === CommunicationConfigurationAction.ARCHIVE) {
            if (config != null) {
                errors.add("The provided event config still present in the source directory")
            }
        } else if (params.action != CommunicationConfigurationAction.SHOW_CURRENT && config == null) {
            errors.add("The provided event config has not been found on the source path")
        }

        if (params.action !in setOf(
                CommunicationConfigurationAction.SHOW,
                CommunicationConfigurationAction.ACTIVATE) && entry == null) {
            errors.add("The provided event does not exist in the db")
        }

        if (params.action in setOf(CommunicationConfigurationAction.ACTIVATE, CommunicationConfigurationAction.ARCHIVE)
            && (params.operator.role != RbacRole.SUPER
                && !CommunicationConfigurationTool.APPROVERS.contains(params.operator.login))) {
            errors.add("${params.action} is not permitted for the current operator")
        }

        return errors
    }

    protected fun getJsonResult(event: T): Result<Any> {
        val result = try {
            JsonUtils.getObjectMapper()
                .writerWithDefaultPrettyPrinter()
                .withDefaultPrettyPrinter()
                .writeValueAsString(event)
        } catch (e: JsonProcessingException) {
            getLogger().error("Failed to prettify json for $event")
            val result: ValidationResult<Any, Defect<*>> =
                ValidationResult.failed("Failed to convert to json", CommonDefects.invalidValue())
            return Result.broken(result)
        }
        return Result.successful(CommunicationConfigurationItem(result, null))
    }

    fun showEntry(config: ParsedCommunicationConfiguration<T>, entry: T?): Result<Any> {
        return getJsonResult(config.configuration)
    }

    fun showDbEntry(config: ParsedCommunicationConfiguration<T>?, entry: T): Result<Any> {
        return getJsonResult(entry)
    }

    fun compareEntry(config: ParsedCommunicationConfiguration<T>, entry: T): Result<Any> {
        val oldJsonResponse = getJsonResult(entry)
        val newJsonResponse = getJsonResult(config.configuration)
        if (!oldJsonResponse.isSuccessful) return oldJsonResponse
        if (!newJsonResponse.isSuccessful) return newJsonResponse

        val newJson = (newJsonResponse.result as CommunicationConfigurationItem).data
        val oldJson = (oldJsonResponse.result as CommunicationConfigurationItem).data

        val diff = DiffUtils.diff(oldJson.lines(), newJson.lines())

        val patch = DiffUtils.generateUnifiedDiff(
            getType().name,
            config.configurationPath.toString(),
            oldJson.lines(),
            diff, 3
        ).joinToString("\n")
        return Result.successful(CommunicationConfigurationItem(newJson, patch))
    }
}
