package ru.yandex.direct.internaltools.tools.communication.service

import java.time.LocalDateTime
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.communication.model.CommunicationEvent
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventsRepository
import ru.yandex.direct.internaltools.tools.communication.model.CommunicationConfigurationParameters
import ru.yandex.direct.internaltools.tools.communication.model.ParsedCommunicationConfiguration
import ru.yandex.direct.internaltools.tools.communication.repository.CommunicationEventConfigurationRepository
import ru.yandex.direct.result.Result
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CommonConstraints.isEqual
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

@Service
class CommunicationEventConfigurationService(
    private val eventsRepository: CommunicationEventsRepository,
    private val eventVersionConfigurationService: CommunicationEventVersionConfigurationService,
    private val configurationRepository: CommunicationEventConfigurationRepository
) : CommunicationConfigurationService<CommunicationEvent>() {
    private val logger = LoggerFactory.getLogger(CommunicationEventConfigurationService::class.java)!!

    override fun getLogger(): Logger = logger
    override fun getType(): Class<CommunicationEvent> {
        return CommunicationEvent::class.java
    }

    override fun getEntryConfigByParams(
        params: CommunicationConfigurationParameters
    ): ParsedCommunicationConfiguration<CommunicationEvent>? {
        val result = configurationRepository.getEvents(params.eventId)
        if (result.hasAnyErrors()) {
            return null
        }
        return result.value
    }

    override fun getEntryByParams(params: CommunicationConfigurationParameters): CommunicationEvent {
        return eventsRepository.getCommunicationEventsById(params.eventId).orElse(null)
    }

    override fun archiveEntry(
        config: ParsedCommunicationConfiguration<CommunicationEvent>?,
        entry: CommunicationEvent?
    ): Result<Any> {
        val hasUnarchivedVersion = eventVersionConfigurationService.getVersionsByEvent(entry!!.eventId)
            .any { it.status != CommunicationEventVersionStatus.ARCHIVED }
        if (hasUnarchivedVersion) {
            val result: ValidationResult<Any, Defect<*>> =
                ValidationResult.failed("The event ${entry.eventId} cannot be archived, " +
                    "because it still has unarchived event versions", CommonDefects.invalidValue())
            return Result.broken(result)
        }

        eventsRepository.archiveCommunicationEvent(entry.eventId, LocalDateTime.now())
        return getJsonResult(eventsRepository.getCommunicationEventsById(entry.eventId).get())
    }

    override fun upsertEntry(
        config: ParsedCommunicationConfiguration<CommunicationEvent>,
        entry: CommunicationEvent?
    ): Result<Any> {
        val id = if (entry == null) {
            eventsRepository.addCommunicationEvent(config.configuration)
        } else {
            val configEvent = config.configuration
            val builder = ItemValidationBuilder.of<CommunicationEvent, Defect<*>>(configEvent)
            builder.item(configEvent.eventId, "eventId")
                .check(isEqual(entry.eventId, CommonDefects.invalidValue()))
            builder.item(configEvent.name, "name")
                .check(isEqual(entry.name, CommonDefects.invalidValue()))
            builder.item(configEvent.type, "type")
                .check(isEqual(entry.type, CommonDefects.invalidValue()))

            val hasErrors = builder.result.hasAnyErrors()

            if (hasErrors) {
                val result: ValidationResult<Any, Defect<*>> =
                    ValidationResult.failed("Event_Id, name and type cannot be changed", CommonDefects.invalidValue())
                return Result.broken(result)
            }

            eventsRepository.updateCommunicationEvent(configEvent)
            config.configuration.eventId
        }
        return getJsonResult(eventsRepository.getCommunicationEventsById(id).get())
    }
}
