package ru.yandex.direct.internaltools.tools.communication.service;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.communication.model.CommunicationEvent;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventStatus;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventType;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventVersionsRepository;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventsRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.internaltools.tools.communication.model.CommunicationEventTypesParameters;
import ru.yandex.direct.internaltools.tools.communication.model.CommunicationEventTypesResponse;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;

import static java.lang.Math.min;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventStatus.ACTIVE;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventStatus.ARCHIVED;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventStatus.NEW_;
import static ru.yandex.direct.internaltools.tools.communication.model.CommunicationEventTypesAction.ARCHIVE;
import static ru.yandex.direct.internaltools.tools.communication.model.CommunicationEventTypesAction.UPDATE;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class CommunicationEventTypesService {

    private final CommunicationEventsRepository communicationEventsRepository;
    private final CommunicationEventVersionsRepository communicationEventVersionsRepository;
    private final UserService userService;

    public static final Map<CommunicationEventStatus, Integer> STATUS_PRIORITY = Map.of(
            ACTIVE, 0,
            NEW_, 1,
            ARCHIVED, 2
    );

    public static final Comparator<CommunicationEvent> EVENTS_COMPARATOR = (event1, event2) ->
            event1.getStatus() == event2.getStatus()
                    ? -Long.compare(event1.getEventId(), event2.getEventId())
                    : Integer.compare(STATUS_PRIORITY.get(event1.getStatus()), STATUS_PRIORITY.get(event2.getStatus()));


    @Autowired
    public CommunicationEventTypesService(CommunicationEventsRepository communicationEventsRepository,
                                          CommunicationEventVersionsRepository communicationEventVersionsRepository,
                                          UserService userService) {
        this.communicationEventsRepository = communicationEventsRepository;
        this.communicationEventVersionsRepository = communicationEventVersionsRepository;
        this.userService = userService;
    }

    public List<CommunicationEvent> getSortedEventsByTypesAndStatuses(
            List<CommunicationEventType> types,
            List<CommunicationEventStatus> statuses
    ) {
        return communicationEventsRepository.getAllCommunicationEvents()
                .stream()
                .filter(event -> types.contains(event.getType()))
                .filter(event -> statuses.contains(event.getStatus()))
                .sorted(EVENTS_COMPARATOR)
                .collect(Collectors.toList());
    }

    public CommunicationEvent updateOwnersAndGetChosenCommunicationEvent(Long eventId, Set<String> eventOwners) {
        communicationEventsRepository.updateCommunicationEventOwners(eventId, eventOwners);
        var events = communicationEventsRepository.getCommunicationEventsByIds(singletonList(eventId));
        return events == null || events.isEmpty() ? null : events.get(0);
    }

    public CommunicationEvent archiveAndGetChosenCommunicationEvent(Long eventId) {
        var archiveTime = StreamEx.of(communicationEventVersionsRepository.getVersionsByEvent(eventId))
                // Для каждого события определяем, когда оно перестало работать (из-за архивации или из-за времени жизни)
                .map(v -> min(v.getStartTime().toEpochSecond(ZoneOffset.UTC), Long.parseLong(v.getLastChangeStatus())))
                // Берем максимум по всем итерациям
                .max(Long::compareTo)
                .map(s -> LocalDateTime.ofEpochSecond(s, 0, ZoneOffset.UTC))
                // Берем текущее время, если итераций не было
                .orElse(LocalDateTime.now());
        communicationEventsRepository.archiveCommunicationEvent(eventId, archiveTime);
        var events = communicationEventsRepository.getCommunicationEventsByIds(singletonList(eventId));
        return events == null || events.isEmpty() ? null : events.get(0);
    }

    public CommunicationEvent addNewEventWithDefaults(
            CommunicationEventType type,
            String eventName,
            Set<String> eventOwners) {
        var event = new CommunicationEvent()
                .withName(eventName)
                .withOwners(eventOwners)
                .withStatus(NEW_)
                .withType(type);
        long eventId = communicationEventsRepository.addCommunicationEvent(event);

        return event
                .withEventId(eventId);
    }

    private CommunicationEventTypesResponse createCommunicationEventTypesResponse(CommunicationEvent communicationEvent) {
        return new CommunicationEventTypesResponse()
                .withEventId(communicationEvent.getEventId())
                .withName(communicationEvent.getName())
                .withOwners(StringUtils.join(communicationEvent.getOwners(), ","))
                .withType(communicationEvent.getType())
                .withStatus(communicationEvent.getStatus())
                .withArchiveTime(communicationEvent.getArchivedTime())
                .withActivationTime(communicationEvent.getActivateTime());
    }

    public List<CommunicationEventTypesResponse> convertToMassResponse(List<CommunicationEvent> communicationEvents) {
        return mapList(communicationEvents, this::createCommunicationEventTypesResponse);
    }

    public Constraint<Set<String>, Defect> loginsExist() {
        return (s) -> {
            var existingLogins = userService.massGetUserByLogin(s)
                    .stream()
                    .map(User::getLogin)
                    .collect(toSet());
            return existingLogins.containsAll(s)
                    ? null
                    : CommonDefects.validLogin();
        };
    }

    /**
     * @return строка, в которой описана ошибка валидации, null при успехе
     */
    public String checkChosenEvent(CommunicationEventTypesParameters params) {
        if (params.getChosenEventId() == null) {
            return "The event must be chosen";
        }
        var eventList = communicationEventsRepository.getCommunicationEventsByIds(singletonList(params.getChosenEventId()));
        if (eventList == null || eventList.isEmpty()) {
            return "No event detected in database";
        }
        var event = eventList.get(0);
        String operatorLogin = params.getOperator().getLogin();
        if (!event.getOwners().contains(operatorLogin)) {
            return "Only owners can change the event";
        }
        if (ARCHIVE == params.getAction() && !allIterationsArchived(params.getChosenEventId())) {
            return "All iterations must be archived";
        }
        if (UPDATE.equals(params.getAction()) && loginsExist().apply(params.getParsedOwnerLogins()) != null) {
            return "Invalid login";
        }
        return null;
    }

    private boolean allIterationsArchived(long eventId) {
        return communicationEventVersionsRepository
                .getVersionsByEvent(eventId)
                .stream()
                .allMatch(version -> version.getStatus() == CommunicationEventVersionStatus.ARCHIVED);
    }
}
