package ru.yandex.direct.internaltools.tools.communication.service

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventVersionsRepository
import ru.yandex.direct.internaltools.tools.communication.model.CommunicationConfigurationParameters
import ru.yandex.direct.internaltools.tools.communication.model.ParsedCommunicationConfiguration
import ru.yandex.direct.internaltools.tools.communication.repository.CommunicationEventConfigurationRepository
import ru.yandex.direct.result.Result
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

@Service
class CommunicationEventVersionConfigurationService(
    private val eventVersionsRepository: CommunicationEventVersionsRepository,
    private val configurationRepository: CommunicationEventConfigurationRepository,
) : CommunicationConfigurationService<CommunicationEventVersion>() {
    private val logger = LoggerFactory.getLogger(CommunicationEventVersionConfigurationService::class.java)

    override fun getLogger(): Logger = logger
    override fun getType(): Class<CommunicationEventVersion> {
        return CommunicationEventVersion::class.java
    }

    fun getVersionsByEvent(eventId: Long): List<CommunicationEventVersion> {
        return eventVersionsRepository.getVersionsByEvent(eventId)
    }

    override fun getEntryConfigByParams(
        params: CommunicationConfigurationParameters
    ): ParsedCommunicationConfiguration<CommunicationEventVersion>? {
        val result = configurationRepository.getEventVersions(params.eventId, params.iteration!!)
        if (result.hasAnyErrors()) {
            return null
        }
        return result.value
    }

    override fun getEntryByParams(params: CommunicationConfigurationParameters): CommunicationEventVersion {
        return eventVersionsRepository.getOptionalVersion(params.eventId, params.iteration!!).orElse(null)
    }

    override fun archiveEntry(
        config: ParsedCommunicationConfiguration<CommunicationEventVersion>?,
        entry: CommunicationEventVersion?
    ): Result<Any> {
        entry!!.status = CommunicationEventVersionStatus.ARCHIVED
        val result = eventVersionsRepository.update(entry, setOf(
            CommunicationEventVersionStatus.NEW_,
            CommunicationEventVersionStatus.ABORTED,
            CommunicationEventVersionStatus.ACTIVE
        ))
        return if (result) {
            getJsonResult(eventVersionsRepository.getVersion(entry.eventId, entry.iter))
        } else {
            val res: ValidationResult<Any, Defect<*>> =
                ValidationResult.failed(
                    "Failed to delete event ${entry.eventId}, ${entry.iter}",
                    CommonDefects.invalidValue())
            return Result.broken(res)
        }
    }

    override fun upsertEntry(
        config: ParsedCommunicationConfiguration<CommunicationEventVersion>,
        entry: CommunicationEventVersion?
    ): Result<Any> {
        val result = if (entry == null) {
            eventVersionsRepository.create(config.configuration)
        } else {
            eventVersionsRepository.update(
                config.configuration,
                CommunicationEventVersionStatus.values().toMutableSet(),
                false
            )
        }

        if (!result) {
            val res: ValidationResult<Any, Defect<*>> =
                ValidationResult.failed(
                    "Failed to create version for ${config.configuration.eventId}",
                    CommonDefects.invalidValue())
            return Result.broken(res)
        }

        val newVersion = eventVersionsRepository.getVersion(config.configuration.eventId, config.configuration.iter)
        return getJsonResult(newVersion)
    }
}
