package ru.yandex.direct.internaltools.tools.communication.util;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Collection;
import java.util.List;
import java.util.function.Predicate;

import ru.yandex.ads.bsyeti.libs.communications.TDirectWebUIData;
import ru.yandex.ads.bsyeti.libs.communications.TEventSource;
import ru.yandex.ads.bsyeti.libs.events.TCommunicationEvent;
import ru.yandex.ads.bsyeti.libs.events.TCommunicationEventData;
import ru.yandex.direct.communication.container.web.Body;
import ru.yandex.direct.communication.container.web.Button;
import ru.yandex.direct.communication.container.web.Footer;
import ru.yandex.direct.communication.container.web.Image;
import ru.yandex.direct.communication.container.web.Message;
import ru.yandex.direct.communication.container.web.Slide;
import ru.yandex.direct.core.entity.communication.model.CommunicationEvent;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventStatus;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventVersionsRepository;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventsRepository;
import ru.yandex.direct.internaltools.tools.communication.model.SendPersonalCommunicationEventParameters;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;

import static java.util.Collections.singletonList;
import static ru.yandex.ads.bsyeti.libs.communications.EChannel.DIRECT_WEB_UI;
import static ru.yandex.ads.bsyeti.libs.communications.ECommunicationType.INFORMATION;
import static ru.yandex.ads.bsyeti.libs.communications.EEventType.SEND_MESSAGE;
import static ru.yandex.ads.bsyeti.libs.communications.ESlot.TOP;
import static ru.yandex.ads.bsyeti.libs.communications.ESourceType.DIRECT_ADMIN;
import static ru.yandex.ads.bsyeti.libs.communications.TDirectWebUIData.EMessageFormat.RICH;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

public class SendPersonalCommunicationEventUtils {

    private static final String ACTION = "action";
    private static final String ACTION_LINK = "actionLink";

    private SendPersonalCommunicationEventUtils() {
    }

    public static TCommunicationEvent createSingleEvent(Long userId, String messageJson) {
        long currentTimeStamp = LocalDateTime.now().toEpochSecond(ZoneOffset.UTC);
        long toTimeStamp = LocalDateTime.now().plusDays(1).toEpochSecond(ZoneOffset.UTC);
        var builder = TCommunicationEvent.newBuilder()
                .addUids(userId)
                .setTimestamp(currentTimeStamp)
                .setData(dataWithDefaultTypes()
                        .setId((int) currentTimeStamp)
                        .setSource(sourceByUserId(userId))
                        .setSendMessage(TCommunicationEventData.TSendMessage.newBuilder()
                                .addChannels(DIRECT_WEB_UI)
                                .setExpired((int) toTimeStamp)
                                .setDirectWebData(directWebDataWithDefaultsAndMessage(messageJson))));
        return builder.build();
    }

    public static String createMessageText(SendPersonalCommunicationEventParameters params) {
        Message currentMessage = new Message().withSlides(
                singletonList(
                        new Slide()
                                .withBody(createBodyByParams(formatText(params.getTitle()),
                                        formatText(params.getText())))
                                .withFooter(createFooterByParams(formatText(params.getButtonText()),
                                        params.getButtonHref()))
                                .withImage(createImageByParams(params.getImageHref()))
                ));
        return JsonUtils.toJson(currentMessage);
    }

    private static TCommunicationEventData.Builder dataWithDefaultTypes() {
        return TCommunicationEventData.newBuilder()
                .setType(SEND_MESSAGE)
                .setCommunicationType(INFORMATION);
    }

    private static TDirectWebUIData.Builder directWebDataWithDefaultsAndMessage(String messageJson) {
        return TDirectWebUIData.newBuilder()
                .setMessageText(messageJson)
                .setPriority(1)
                .setMessageFormat(RICH)
                .setSlot(TOP)
                .setMessageText(messageJson);
    }

    private static TEventSource sourceByUserId(Long userId) {
        return TEventSource.newBuilder()
                .setType(DIRECT_ADMIN)
                .setId(userId)
                .build();
    }

    private static Body createBodyByParams(String title, String text) {
        return title != null || text != null
                ? new Body()
                .withTitle(title)
                .withText(text)
                : null;
    }

    private static Image createImageByParams(String imageUrl) {
        return imageUrl != null ? new Image().withUrl(imageUrl) : null;
    }

    private static Footer createFooterByParams(String buttonText, String buttonHref) {
        if (buttonHref != null || buttonText != null) {
            return new Footer().withButtons(singletonList(
                    new Button()
                            .withId(1)
                            .withView(ACTION)
                            .withAction(ACTION_LINK)
                            .withHref(buttonHref)
                            .withText(buttonText)
                            .withInLine(false)
            ));
        }
        return null;
    }

    private static String formatText(String text) {
        return text == null ? null : text.replace("\\n", "\n");
    }

    /**
     * Проверяет номер итерации события
     *
     * @param versionsRepository репозиторий итераций
     * @param eventId            идентификатор события
     * @return INVALID_VALUE если не найдена итерация
     */
    public static Validator<Long, Defect> getEventIterationValidator(CommunicationEventVersionsRepository versionsRepository, Long eventId) {
        return iteration -> {
            ItemValidationBuilder<Long, Defect> builder = ItemValidationBuilder.of(iteration);
            builder.check(notNull());
            builder.checkByFunction(
                    iter -> versionsRepository
                            .getOptionalVersion(eventId, iter).isEmpty()
                            ? invalidValue()
                            : null,
                    When.isValid());
            return builder.getResult();
        };
    }

    /**
     * Проверяет идентификатор события
     *
     * @param eventsRepository репозиторий событий
     * @param invalidStatuses  список невалидных статусов
     * @return INVALID_VALUE если не найдено событие в валидном статусе
     */
    public static Validator<Long, Defect> getEventIdValidator(CommunicationEventsRepository eventsRepository,
                                                              Collection<CommunicationEventStatus> invalidStatuses) {
        return eventId -> {
            ItemValidationBuilder<Long, Defect> builder = ItemValidationBuilder.of(eventId);
            builder.check(notNull());
            builder.checkByFunction(id -> eventsRepository
                            .getCommunicationEventsByIds(List.of(id))
                            .stream()
                            .map(CommunicationEvent::getStatus)
                            .anyMatch(Predicate.not(invalidStatuses::contains))
                            ? null
                            : invalidValue(),
                    When.isValid());
            return builder.getResult();
        };
    }
}
