package ru.yandex.direct.internaltools.tools.contentcategories;

import java.util.List;
import java.util.Objects;

import com.google.common.base.Strings;

import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.contentcategories.model.ContentCategoriesWhitelistRecord;
import ru.yandex.direct.internaltools.tools.contentcategories.model.ContentCategoriesWhitelistUpdateParams;
import ru.yandex.direct.internaltools.tools.contentcategories.repository.ContentCategoriesWhitelistYtRepository;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.joining;
import static ru.yandex.direct.internaltools.tools.contentcategories.model.ContentCategoriesWhitelistUpdateParams.MINUS_CATEGORIES_LABEL;
import static ru.yandex.direct.internaltools.tools.contentcategories.model.ContentCategoriesWhitelistUpdateParams.PLUS_CATEGORIES_LABEL;
import static ru.yandex.direct.internaltools.tools.contentcategories.model.ContentCategoriesWhitelistUpdateParams.URL_PATTERN_LABEL;
import static ru.yandex.direct.utils.StringUtils.parseLongsFromString;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Tool(
        name = "Редактирование whitelist-а для категорий и жанров",
        label = "content_categories_whitelist",
        description = "Позволяет редактировать whitelist, " +
                "который применяется на финальном этапе разметки страниц категориями контента и brandsafety",
        consumes = ContentCategoriesWhitelistUpdateParams.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.OTHER)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.SUPERREADER, InternalToolAccessRole.DEVELOPER})
public class ContentCategoriesWhitelistTool extends
        MassInternalTool<ContentCategoriesWhitelistUpdateParams, ContentCategoriesWhitelistRecord> {

    private final ContentCategoriesWhitelistYtRepository contentCategoriesWhitelistYtRepository;

    public ContentCategoriesWhitelistTool(
            ContentCategoriesWhitelistYtRepository contentCategoriesWhitelistYtRepository
    ) {
        this.contentCategoriesWhitelistYtRepository = contentCategoriesWhitelistYtRepository;
    }

    @Override
    public ValidationResult<ContentCategoriesWhitelistUpdateParams, Defect> validate(
            ContentCategoriesWhitelistUpdateParams parameters) {
        ItemValidationBuilder<ContentCategoriesWhitelistUpdateParams, Defect> validationBuilder =
                ItemValidationBuilder.of(parameters);

        validationBuilder.item(parameters.getUrlPattern(), URL_PATTERN_LABEL).check(notBlank());
        if (ContentCategoriesWhitelistUpdateParams.ChangeAction.ADD_OR_UPDATE_PATTERN.equals(parameters.getChangeAction())
                && Strings.isNullOrEmpty(parameters.getMinusCategories())) {
            validationBuilder.item(parameters.getPlusCategories(), PLUS_CATEGORIES_LABEL).check(notNull());
        }

        validationBuilder.item(parameters.getPlusCategories(), PLUS_CATEGORIES_LABEL).check(this::validateCategories);
        validationBuilder.item(parameters.getMinusCategories(), MINUS_CATEGORIES_LABEL).check(this::validateCategories);

        return validationBuilder.getResult();
    }

    @Override
    protected List<ContentCategoriesWhitelistRecord> getMassData(
            ContentCategoriesWhitelistUpdateParams parameters) {
        parameters.setUrlPattern(parameters.getUrlPattern().trim());

        switch (parameters.getChangeAction()) {

            case ADD_OR_UPDATE_PATTERN:
                var plusCategories = formatCategories(parameters.getPlusCategories());
                var minusCategories = formatCategories(parameters.getMinusCategories());
                contentCategoriesWhitelistYtRepository.insertOrUpdate(
                        parameters.getUrlPattern(), plusCategories, minusCategories);
                break;

            case DELETE_PATTERN:
                contentCategoriesWhitelistYtRepository.delete(parameters.getUrlPattern());
                break;
        }

        return contentCategoriesWhitelistYtRepository.getAll();
    }

    @Override
    protected List<ContentCategoriesWhitelistRecord> getMassData() {
        return contentCategoriesWhitelistYtRepository.getAll();
    }

    private static String formatCategories(String categories) {
        if (Strings.isNullOrEmpty(categories)) {
            return null;
        }

        return parseLongsFromString(categories)
                .stream()
                .filter(Objects::nonNull)
                .map(String::valueOf)
                .collect(joining(","));
    }

    private Defect validateCategories(String categories) {
        if (Strings.isNullOrEmpty(categories)) {
            return null;
        }

        var allValid = parseLongsFromString(categories).stream()
                // it's either content category, or brand safety, which is a power of two
                .allMatch(v -> GoalType.CONTENT_CATEGORY.equals(Goal.computeType(v)) || ((v & (v - 1)) == 0));
        return allValid ? null : CommonDefects.invalidValue();
    }
}
