package ru.yandex.direct.internaltools.tools.contentcategories.repository;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.grid.schema.yt.tables.CategoriesBlockedDomains;
import ru.yandex.direct.internaltools.tools.contentcategories.model.ContentCategoriesBlockedDomainsRecord;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.ytcomponents.config.DirectYtDynamicConfig;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.grid.schema.yt.Tables.CATEGORIES_BLOCKED_DOMAINS;
import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;

@Repository
@ParametersAreNonnullByDefault
public class ContentCategoriesBlockedDomainsYtRepository {
    private static final Logger logger = LoggerFactory.getLogger(ContentCategoriesBlockedDomainsYtRepository.class);
    private static final CategoriesBlockedDomains CATEGORIES_BLOCKED_DOMAINS_ALIAS = CATEGORIES_BLOCKED_DOMAINS.as("D");
    private static final Field<String> DOMAIN = aliased(CATEGORIES_BLOCKED_DOMAINS_ALIAS.DOMAIN);

    private static final TableSchema TABLE_SCHEMA = new TableSchema.Builder()
            .addKey("domain", ColumnValueType.STRING)
            .addValue("value", ColumnValueType.STRING)
            .build();

    private static final String TRACE_TAG = "content_categories:yql";

    private final YtProvider ytProvider;
    private final YtCluster ytCluster;
    private final String tablePath;

    public ContentCategoriesBlockedDomainsYtRepository(
            YtProvider ytProvider,
            DirectYtDynamicConfig ytConfig
    ) {
        this.ytProvider = ytProvider;
        this.ytCluster = ytConfig.getCategoriesClusters().iterator().next();
        this.tablePath = ytConfig.tables().direct().categoriesBlockedDomainsTablePath();
    }

    public List<ContentCategoriesBlockedDomainsRecord> getAll() {
        var query = YtDSL.ytContext()
                .select(DOMAIN)
                .from(CATEGORIES_BLOCKED_DOMAINS_ALIAS);

        try (var ignore = Trace.current().profile(TRACE_TAG, "get_all_blocked_domains_records")) {
            try {
                return ytProvider.getDynamicOperator(ytCluster)
                        .selectRows(query)
                        .getYTreeRows()
                        .stream()
                        .map(row -> new ContentCategoriesBlockedDomainsRecord().withDomain(row.getString(DOMAIN.getName())))
                        .collect(toList());
            } catch (RuntimeException ex) {
                logger.error("Error while getting all blocked domains records", ex);
                return emptyList();
            }
        }
    }

    public void insert(Set<String> domains) {
        try (var ignore = Trace.current().profile(TRACE_TAG, "insert_blocked_domains_record")) {
            try {
                var dynamicOperator = ytProvider.getDynamicOperator(ytCluster);
                var request = new ModifyRowsRequest(tablePath, TABLE_SCHEMA);
                domains.forEach(domain -> request.addInsert(List.of(domain, "")));
                dynamicOperator.runInTransaction(tx -> tx.modifyRows(request).join()); // IGNORE-BAD-JOIN DIRECT-149116
            } catch (RuntimeException ex) {
                logger.error("Error while inserting blocked domains record for " + domains, ex);
            }
        }
    }

    public void delete(Set<String> domains) {
        try (var ignore = Trace.current().profile(TRACE_TAG, "delete_blocked_domains_record")) {
            try {
                var dynamicOperator = ytProvider.getDynamicOperator(ytCluster);
                var request = new ModifyRowsRequest(tablePath, TABLE_SCHEMA);
                domains.forEach(domain -> request.addDelete(singletonList(domain)));
                dynamicOperator.runInTransaction(tx -> tx.modifyRows(request).join()); // IGNORE-BAD-JOIN DIRECT-149116
            } catch (RuntimeException ex) {
                logger.error("Error while deleting blocked domains record for " + domains, ex);
            }
        }
    }
}
