package ru.yandex.direct.internaltools.tools.contentcategories.repository;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.grid.schema.yt.tables.WhitelistCategories;
import ru.yandex.direct.internaltools.tools.contentcategories.model.ContentCategoriesWhitelistRecord;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.ytcomponents.config.DirectYtDynamicConfig;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.grid.schema.yt.tables.WhitelistCategories.WHITELIST_CATEGORIES;
import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;

@Repository
@ParametersAreNonnullByDefault
public class ContentCategoriesWhitelistYtRepository {
    private static final Logger logger = LoggerFactory.getLogger(ContentCategoriesWhitelistYtRepository.class);
    private static final WhitelistCategories WHITELIST_CATEGORIES_ALIAS = WHITELIST_CATEGORIES.as("B");
    private static final Field<String> URL_PATTERN = aliased(WHITELIST_CATEGORIES_ALIAS.URL_PATTERN);
    private static final Field<String> PLUS_CATEGORIES = aliased(WHITELIST_CATEGORIES_ALIAS.PLUS_CATEGORIES);
    private static final Field<String> MINUS_CATEGORIES = aliased(WHITELIST_CATEGORIES_ALIAS.MINUS_CATEGORIES);

    private static final TableSchema TABLE_SCHEMA = new TableSchema.Builder()
            .addKey("url_pattern", ColumnValueType.STRING)
            .addValue("plus_categories", ColumnValueType.STRING)
            .addValue("minus_categories", ColumnValueType.STRING)
            .build();
    private static final String TRACE_TAG = "content_categories:yql";

    private final YtProvider ytProvider;
    private final YtCluster ytCluster;
    private final String tablePath;

    public ContentCategoriesWhitelistYtRepository(
            YtProvider ytProvider,
            DirectYtDynamicConfig ytConfig
    ) {
        this.ytProvider = ytProvider;
        this.ytCluster = ytConfig.getCategoriesClusters().iterator().next();
        this.tablePath = ytConfig.tables().direct().categoriesWhitelistTablePath();
    }

    public List<ContentCategoriesWhitelistRecord> getAll() {
        var query = YtDSL.ytContext()
                .select(URL_PATTERN, PLUS_CATEGORIES, MINUS_CATEGORIES)
                .from(WHITELIST_CATEGORIES_ALIAS);

        try (var ignore = Trace.current().profile(TRACE_TAG, "get_all_whitelist_records")) {
            try {
                return ytProvider.getDynamicOperator(ytCluster)
                        .selectRows(query)
                        .getYTreeRows()
                        .stream()
                        .map(this::readWhitelistRecord)
                        .collect(toList());
            } catch (RuntimeException ex) {
                logger.error("Error while getting all whitelist records", ex);
                return emptyList();
            }
        }
    }

    public void insertOrUpdate(String urlPattern, @Nullable String plusCategories, @Nullable String minusCategories) {
        try (var ignore = Trace.current().profile(TRACE_TAG, "insert_whitelist_record")) {
            try {
                var dynamicOperator = ytProvider.getDynamicOperator(ytCluster);
                var request = new ModifyRowsRequest(tablePath, TABLE_SCHEMA)
                        .addInsert(asList(urlPattern, plusCategories, minusCategories));

                dynamicOperator.runInTransaction(tx -> tx.modifyRows(request).join()); // IGNORE-BAD-JOIN DIRECT-149116
            } catch (RuntimeException ex) {
                logger.error("Error while inserting whitelist record for " + urlPattern, ex);
            }
        }
    }

    public void delete(String urlPattern) {
        try (var ignore = Trace.current().profile(TRACE_TAG, "delete_whitelist_record")) {
            try {
                var dynamicOperator = ytProvider.getDynamicOperator(ytCluster);
                var request = new ModifyRowsRequest(tablePath, TABLE_SCHEMA).addDelete(singletonList(urlPattern));

                dynamicOperator.runInTransaction(tx -> tx.modifyRows(request).join()); // IGNORE-BAD-JOIN DIRECT-149116
            } catch (RuntimeException ex) {
                logger.error("Error while deleting whitelist record for " + urlPattern, ex);
            }
        }
    }

    private ContentCategoriesWhitelistRecord readWhitelistRecord(YTreeMapNode row) {
        return new ContentCategoriesWhitelistRecord()
                .withUrlPattern(row.getString(URL_PATTERN.getName()))
                .withPlusCategories(row.getStringO(PLUS_CATEGORIES.getName()).orElse(null))
                .withMinusCategories(row.getStringO(MINUS_CATEGORIES.getName()).orElse(null));
    }
}
