package ru.yandex.direct.internaltools.tools.conversionsourcetype.tool;

import java.io.IOException;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.avatars.client.exception.AvatarsClientCommonException;
import ru.yandex.direct.avatars.client.model.AvatarId;
import ru.yandex.direct.core.entity.conversionsourcetype.model.ConversionSourceType;
import ru.yandex.direct.core.entity.conversionsourcetype.service.ConversionSourceTypeService;
import ru.yandex.direct.core.entity.freelancer.service.AvatarsClientPool;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolMassResult;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.conversionsourcetype.model.ConversionSourceTypeInput;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.freelancer.service.validation.AvatarsDefects.serverConnectionProblem;
import static ru.yandex.direct.core.entity.freelancer.service.validation.AvatarsDefects.unknownError;

@Tool(
        consumes = ConversionSourceTypeInput.class,
        description = "Позволяет создавать, изменять и удалять типы источников конверсий. Для операции удаления " +
                "учитывается ТОЛЬКО поле ID шаблона. Для операций создания и изменения под каждым полем " +
                "подписано, как оно используется.",
        label = "manage_conversion_source_types",
        name = "Управление типами источников конверсий",
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@Category(InternalToolCategory.OTHER)
@ParametersAreNonnullByDefault
public class ConversionSourceTypeManagingTool implements BaseInternalTool<ConversionSourceTypeInput> {

    static final String CREATING_OPERATION_STRING = "creating";
    static final String UPDATING_OPERATION_STRING = "updating";
    static final String REMOVING_OPERATION_STRING = "removing";
    static final String OPERATION_SUCCESSFUL_STRING = "Conversion source type %d has been successfully %s";
    static final String OPERATION_WITH_ERRORS_STRING = "The following errors occur while %s conversion source type " +
            "%d: %s";
    static final String OPERATION_WITH_VALIDATION_ERRORS_STRING = "The following validation errors occur while %s " +
            "conversion source type %d: %s";
    static final String ICON_UPLOADING_ERROR = "The following errors occur while %s conversion source type %d: %s";

    private final AvatarsClientPool conversionSourceTypeAvatarsClientPool;

    private final ConversionSourceTypeService conversionSourceTypeService;

    @Autowired
    public ConversionSourceTypeManagingTool(
            ConversionSourceTypeService conversionSourceTypeService,
            AvatarsClientPool freelancersAvatarsClientPool
    ) {
        this.conversionSourceTypeService = conversionSourceTypeService;
        this.conversionSourceTypeAvatarsClientPool = freelancersAvatarsClientPool;
    }

    @Override
    public InternalToolResult processWithoutInput() {
        var conversionSourceTypes = conversionSourceTypeService.getAll();

        var conversionSourceTypeInfoList = conversionSourceTypes.stream()
                .map(ConversionSourceTypeConverter::fromConversionSourceType)
                .collect(Collectors.toList());

        return new InternalToolMassResult<>(conversionSourceTypeInfoList);
    }

    @Override
    public InternalToolResult process(ConversionSourceTypeInput input) {
        var action = input.getAction();
        switch (action) {
            case CREATE:
                return create(input);
            case UPDATE:
                return update(input);
            case REMOVE:
                return remove(input);
            default:
                return new InternalToolResult("Action " + action + " is not implemented yet");
        }
    }

    private InternalToolResult create(ConversionSourceTypeInput input) {
        ConversionSourceType conversionSourceType;
        if (input.getIcon() != null) {
            var uploadIconResult = uploadImageAndReturnUrl(input.getIcon());
            if (!uploadIconResult.isSuccessful()) {
                return new InternalToolResult(String.format(ICON_UPLOADING_ERROR, CREATING_OPERATION_STRING, 0,
                        uploadIconResult.getErrors().toString()));
            }
            conversionSourceType = ConversionSourceTypeConverter.toConversionSourceType(input,
                    uploadIconResult.getResult());
        } else {
            conversionSourceType = ConversionSourceTypeConverter.toConversionSourceType(input);
        }

        var result = conversionSourceTypeService.add(singletonList(conversionSourceType));
        return processResult(result, 0L, CREATING_OPERATION_STRING);
    }

    private InternalToolResult update(ConversionSourceTypeInput input) {
        var id = input.getId();
        String iconUrl = null;

        if (input.getIcon() != null) {
            var uploadIconResult = uploadImageAndReturnUrl(input.getIcon());
            if (!uploadIconResult.isSuccessful()) {
                return new InternalToolResult(String.format(ICON_UPLOADING_ERROR, UPDATING_OPERATION_STRING, id,
                        uploadIconResult.getErrors().toString()));
            }
            iconUrl = uploadIconResult.getResult();
        }

        var modelChange = new ModelChanges<>(id, ConversionSourceType.class)
                .processNotNull(input.getName(), ConversionSourceType.NAME)
                .processNotNull(input.getDescription(),
                        ConversionSourceType.DESCRIPTION)
                .processNotNull(input.getIsDraft(),
                        ConversionSourceType.IS_DRAFT)
                .processNotNull(input.getPosition(),
                        ConversionSourceType.POSITION)
                .processNotNull(input.getCode(),
                        ConversionSourceType.CODE)
                .processNotNull(input.getIsEditable(),
                        ConversionSourceType.IS_EDITABLE)
                .processNotNull(input.getNameEn(), ConversionSourceType.NAME_EN)
                .processNotNull(input.getDescriptionEn(),
                        ConversionSourceType.DESCRIPTION_EN);

        if (input.getUnsetIconUrl()) {
            modelChange.process(null, ConversionSourceType.ICON_URL);
        } else {
            modelChange.processNotNull(iconUrl, ConversionSourceType.ICON_URL);
        }

        if (input.getUnsetActivationUrl()) {
            modelChange.process(null, ConversionSourceType.ACTIVATION_URL);
        } else {
            modelChange.processNotNull(input.getActivationUrl(),
                    ConversionSourceType.ACTIVATION_URL);
        }

        var result = conversionSourceTypeService.update(singletonList(modelChange));

        return processResult(result, id, UPDATING_OPERATION_STRING);

    }

    private InternalToolResult remove(ConversionSourceTypeInput input) {
        var id = input.getId();
        var result = conversionSourceTypeService.remove(singletonList(id));
        return processResult(result, id, REMOVING_OPERATION_STRING);
    }

    private InternalToolResult processResult(MassResult<Long> result, Long id, String operationType) {
        if (!result.isSuccessful()) {
            return new InternalToolResult(String.format(OPERATION_WITH_ERRORS_STRING, operationType, id,
                    result.getErrors().toString()));
        }
        if (!result.getValidationResult().flattenErrors().isEmpty()) {
            return new InternalToolResult(String.format(OPERATION_WITH_VALIDATION_ERRORS_STRING, operationType, id,
                    result.getValidationResult().flattenErrors().toString()));
        }
        return new InternalToolResult(String.format(OPERATION_SUCCESSFUL_STRING, id, operationType));
    }

    private Result<String> uploadImageAndReturnUrl(byte[] image) {
        var avatarsClient = conversionSourceTypeAvatarsClientPool.getDefaultClient();
        AvatarId result;
        try {
            result = avatarsClient.upload(image);
        } catch (AvatarsClientCommonException ex) {
            if (ex.getCause() instanceof IOException) {
                ValidationResult<String, Defect> failedValidationResult =
                        ValidationResult.failed("", serverConnectionProblem());
                return Result.broken(failedValidationResult);
            }
            return Result.broken(ValidationResult.failed("", unknownError()));
        }

        var url = "https://avatars.mdst.yandex.net/get-" + result.toAvatarIdString() + "/orig";

        return Result.successful(url);
    }
}
