package ru.yandex.direct.internaltools.tools.cpm;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithBrandLift;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignWithBrandLiftExperimentsService;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.measurers.model.CpmCampOperationInfo;
import ru.yandex.direct.internaltools.tools.measurers.model.CpmCampToolParameter;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.getLongIdsFromString;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.isStringWithValidLongIds;

@Tool(
        name = "Перепослать эксперименты для РК с брендлифтом",
        label = "resent_bl_exp",
        description = "Полезно чтобы флаги bkSettings учитывались",
        consumes = CpmCampToolParameter.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.CPM)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class BlExpResentTool extends MassInternalTool<CpmCampToolParameter, CpmCampOperationInfo> {
    private final ShardHelper shardHelper;
    private final CampaignWithBrandLiftExperimentsService experimentsService;
    private final CampaignTypedRepository campaignTypedRepository;
    private final RetargetingConditionService retargetingConditionService;


    public BlExpResentTool(ShardHelper shardHelper,
                           CampaignWithBrandLiftExperimentsService experimentsService,
                           CampaignTypedRepository campaignTypedRepository,
                           RetargetingConditionService retargetingConditionService) {
        this.shardHelper = shardHelper;
        this.experimentsService = experimentsService;
        this.campaignTypedRepository = campaignTypedRepository;
        this.retargetingConditionService = retargetingConditionService;
    }

    @Override
    protected List<CpmCampOperationInfo> getMassData(CpmCampToolParameter parameter) {
        ArrayList<CpmCampOperationInfo> res = new ArrayList<>();
        shardHelper.groupByShard(getLongIdsFromString(parameter.getCampaignIds()), ShardKey.CID)
                .forEach((shard, list) -> res.addAll(resent(shard, list)));
        return res;
    }

    private List<CpmCampOperationInfo> resent(Integer shard, List<Long> cids) {
        ArrayList<CpmCampOperationInfo> res = new ArrayList<>(cids.size());
        List<CampaignWithBrandLift> typedCampaigns = (List<CampaignWithBrandLift>) campaignTypedRepository.getTypedCampaigns(shard, cids);
        for (CampaignWithBrandLift camp : typedCampaigns) {
            var login = shardHelper.getLoginByUid(camp.getUid());
            Long abSegmentRetargetingConditionId = camp.getAbSegmentRetargetingConditionId();
            RetargetingConditionBase retargetingCondition =
                    retargetingConditionService.get(ClientId.fromLong(camp.getClientId()), camp.getUid(),
                            List.of(abSegmentRetargetingConditionId)).get(0);
            var rule = retargetingCondition.getRules().get(0);//для внутреннего инструмента такой стиль норм, пятисоток не боимся
            camp.setExperimentId(rule.getSectionId());
            experimentsService.renameExperiment(login, camp);
            res.add(new CpmCampOperationInfo()
                            .setId(camp.getId())
                            .setDesc("exp="+camp.getExperimentId()+", login="+login));
        }
        return res;
    }

    @Override
    public ValidationResult<CpmCampToolParameter, Defect> validate(CpmCampToolParameter parameter) {
        ItemValidationBuilder<CpmCampToolParameter, Defect> validationBuilder =
                ItemValidationBuilder.of(parameter);
        validationBuilder
                .item(parameter.getCampaignIds(), "campaignIds")
                .check(isStringWithValidLongIds(1));
        return validationBuilder.getResult();
    }
}
