package ru.yandex.direct.internaltools.tools.cpm

import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.service.GrutUacBannerService
import ru.yandex.direct.core.entity.uac.service.UacBannerService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.internaltools.core.BaseInternalTool
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup
import ru.yandex.direct.internaltools.core.annotations.tool.Action
import ru.yandex.direct.internaltools.core.annotations.tool.Category
import ru.yandex.direct.internaltools.core.annotations.tool.Tool
import ru.yandex.direct.internaltools.core.container.InternalToolResult
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole
import ru.yandex.direct.internaltools.core.enums.InternalToolAction
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory
import ru.yandex.direct.internaltools.core.enums.InternalToolType
import ru.yandex.direct.internaltools.tools.measurers.model.CpmCampToolParameter
import ru.yandex.direct.internaltools.utils.ToolParameterUtils
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import javax.annotation.ParametersAreNonnullByDefault

@Tool(
    name = "Синхронизировать uac cpm объявления",
    label = "cpm_uac_sync_tool",
    description = "Для списка кампаний поставить в очередь задачу синхронизации объявлений из грута в MySQL.",
    consumes = CpmCampToolParameter::class,
    type = InternalToolType.WRITER
)
@Action(InternalToolAction.REFRESH)
@Category(InternalToolCategory.CPM)
@AccessGroup(InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER)
@ParametersAreNonnullByDefault
class CpmUacSyncTool(
    private val uacBannerService: UacBannerService,
    private val shardHelper: ShardHelper,
    private val grutTransactionProvider: GrutTransactionProvider,
    private val grutUacBannerService: GrutUacBannerService,
) : BaseInternalTool<CpmCampToolParameter> {
    override fun validate(param: CpmCampToolParameter): ValidationResult<CpmCampToolParameter, Defect<Any>> {
        val vb: ItemValidationBuilder<CpmCampToolParameter, Defect<Any>> = ItemValidationBuilder.of(param)
        vb.item(param.campaignIds, "Список ID кампаний")
            .check(ToolParameterUtils.isStringWithValidLongIds(1))
            .check(CommonConstraints.notNull())
        return vb.result
    }

    override fun process(param: CpmCampToolParameter): InternalToolResult {
        val cids = ToolParameterUtils.getLongIdsFromString(param.campaignIds)
        val clientIdByDirectCampaignId = shardHelper.getClientIdsByCampaignIds(cids)
        val notFoundCampaignIds = cids.minus(clientIdByDirectCampaignId.keys)
        if (notFoundCampaignIds.isNotEmpty()) {
            return InternalToolResult().withMessage("Not found " + notFoundCampaignIds.joinToString(","))
        }
        clientIdByDirectCampaignId
            .forEach { (cid, clientId) ->
                grutUacBannerService.updateBriefSynced(cid.toString(), false)
                uacBannerService.updateAdsDeferred(ClientId.fromLong(clientId), param.operator.uid, cid.toString())
            }

        return InternalToolResult().withMessage("Успешно")
    }
}
