package ru.yandex.direct.internaltools.tools.deal;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.deal.service.DealService;
import ru.yandex.direct.dbutil.model.LoginAndClientId;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.deal.container.SearchDealsInfo;
import ru.yandex.direct.internaltools.tools.deal.model.SearchDealsParameters;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Comparator.comparingLong;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.getLongIdsFromString;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.isStringWithValidLongIds;

@Tool(
        name = "Поиск владельцев сделок",
        label = "search_deals",
        description = "Находит и выводит владельцев сделок",
        consumes = SearchDealsParameters.class,
        type = InternalToolType.REPORT
)
@Action(InternalToolAction.SHOW)
@Category(InternalToolCategory.DEALS)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.SUPERREADER, InternalToolAccessRole.SUPPORT})
@ParametersAreNonnullByDefault
public class SearchDealsTool extends MassInternalTool<SearchDealsParameters, SearchDealsInfo> {

    private final DealService dealService;

    @Autowired
    public SearchDealsTool(DealService dealService) {
        this.dealService = dealService;
    }


    @Override
    public ValidationResult<SearchDealsParameters, Defect> validate(
            SearchDealsParameters parameters) {
        ItemValidationBuilder<SearchDealsParameters, Defect> validationBuilder = ItemValidationBuilder.of(parameters);

        validationBuilder
                .item(parameters.getDealIds(), "dealIds")
                .check(isStringWithValidLongIds(1));
        return validationBuilder.getResult();
    }

    @Override
    protected List<SearchDealsInfo> getMassData(SearchDealsParameters parameters) {
        Set<Long> dealIds = getLongIdsFromString(parameters.getDealIds());

        Map<Long, LoginAndClientId> clientsByDealIds = dealService.getLoginAndClientIdsByDealIds(dealIds);

        return EntryStream.of(clientsByDealIds)
                .map(c -> new SearchDealsInfo()
                        .withDealId(c.getKey())
                        .withClientId(c.getValue().getClientId().asLong())
                        .withLogin(c.getValue().getLogin())
                )
                //Сортируем по dealId
                .sorted(comparingLong(SearchDealsInfo::getDealId))
                .collect(Collectors.toList());
    }
}
