package ru.yandex.direct.internaltools.tools.ecom

import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.uac.service.EcomDomainsService
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup
import ru.yandex.direct.internaltools.core.annotations.tool.Action
import ru.yandex.direct.internaltools.core.annotations.tool.Category
import ru.yandex.direct.internaltools.core.annotations.tool.Tool
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole
import ru.yandex.direct.internaltools.core.enums.InternalToolAction
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory
import ru.yandex.direct.internaltools.core.enums.InternalToolType
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool
import ru.yandex.direct.internaltools.tools.ecom.model.CheckLinkInfo
import ru.yandex.direct.internaltools.tools.ecom.model.CheckLinkInfoParam
import ru.yandex.direct.internaltools.utils.ToolParameterUtils.parseCommaSeparatedString
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.constraint.StringConstraints
import ru.yandex.direct.validation.util.validateObject
import java.time.Duration
import java.time.LocalDateTime

private const val DESCRIPTION =
    "По списку урлов с протоколами через запятую отдает данные является ли домен екомовым, " +
        "есть ли он в черном списке или списке маркетплейов/соцсетей, " +
        "сколько офферов мы там нашли, есть ли и когда в последний раз обновлялось превью."

@Tool(
    name = "Проверка еком свойств домена",
    label = "check_link_info_result",
    description = DESCRIPTION,
    consumes = CheckLinkInfoParam::class,
    type = InternalToolType.REPORT
)
@Action(InternalToolAction.SHOW)
@Category(InternalToolCategory.ECOM)
@AccessGroup(
    InternalToolAccessRole.SUPER,
    InternalToolAccessRole.DEVELOPER,
    InternalToolAccessRole.SUPERREADER,
    InternalToolAccessRole.MANAGER,
    InternalToolAccessRole.SUPPORT
)
class CheckLinkInfoResultTool(
    val ecomDomainsService: EcomDomainsService,
    val ppcPropertiesSupport: PpcPropertiesSupport
) : MassInternalTool<CheckLinkInfoParam, CheckLinkInfo>() {
    override fun getMassData(parameter: CheckLinkInfoParam): MutableList<CheckLinkInfo> {
        val notAllowedDomains = ppcPropertiesSupport
            .get<Set<String>>(PpcPropertyNames.DOMAINS_NOT_ALLOWED_FOR_FEED_FROM_SITE, Duration.ofMinutes(5))
            .getOrDefault(emptySet())

        val urls = parseCommaSeparatedString(parameter.urls)
        return urls.map { checkUrl(it, notAllowedDomains) }.toMutableList()
    }

    private fun checkUrl(url: String, notAllowedDomains: Set<String>): CheckLinkInfo {
        if (validateObject(url) {
                check(CommonConstraints.notNull())
                check(StringConstraints.notBlank())
                check(StringConstraints.validHref())
            }.hasAnyErrors()) {
            return CheckLinkInfo(
                url = url,
                isValid = false,
                domain = "",
                isEcom = false,
                isMarketplace = false,
                isInBlacklist = false,
                hasPreview = false,
                strongOffersCount = 0L,
                lastPreviewUpdateDate = null
            )
        }

        val domain = ecomDomainsService.toMainMirrors(listOf(url))[url] ?: url
        val isInBlacklist = notAllowedDomains.contains(domain)
        var hasPreview = false
        var strongOffersCount = 0L
        var lastPreviewUpdateDate: LocalDateTime? = null
        val ecomDomains = ecomDomainsService.getEcomDomainsByDomains(listOf(domain))
        if (ecomDomains.isNotEmpty()) {
            strongOffersCount = ecomDomains[0].offersCount
            hasPreview = ecomDomains[0].previewOffers?.isNotBlank() ?: false
            lastPreviewUpdateDate = ecomDomains[0].lastModifiedPreview
        }

        val isMarketplace = ecomDomainsService.getMarketplacesByDomains(listOf(domain)).isNotEmpty()

        return CheckLinkInfo(
            url = url,
            isValid = true,
            domain = domain,
            isEcom = ecomDomains.isNotEmpty(),
            isMarketplace = isMarketplace,
            isInBlacklist = isInBlacklist,
            hasPreview = hasPreview,
            strongOffersCount = strongOffersCount,
            lastPreviewUpdateDate = lastPreviewUpdateDate
        )
    }
}
