package ru.yandex.direct.internaltools.tools.ecom

import ru.yandex.direct.core.entity.uac.model.EcomDomain
import ru.yandex.direct.core.entity.uac.repository.mysql.EcomDomainsRepository
import ru.yandex.direct.core.entity.uac.service.EcomDomainsService
import ru.yandex.direct.gemini.GeminiClient
import ru.yandex.direct.internaltools.core.BaseInternalTool
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup
import ru.yandex.direct.internaltools.core.annotations.tool.Action
import ru.yandex.direct.internaltools.core.annotations.tool.Category
import ru.yandex.direct.internaltools.core.annotations.tool.Tool
import ru.yandex.direct.internaltools.core.container.InternalToolMassResult
import ru.yandex.direct.internaltools.core.container.InternalToolResult
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole
import ru.yandex.direct.internaltools.core.enums.InternalToolAction
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory
import ru.yandex.direct.internaltools.core.enums.InternalToolType
import ru.yandex.direct.internaltools.tools.ecom.model.AddEcomDomainParam
import ru.yandex.direct.internaltools.tools.ecom.model.EcomDomainAction
import ru.yandex.direct.internaltools.tools.ecom.model.EcomDomainInfo
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.utils.model.UrlParts
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.constraint.StringConstraints.notBlank
import ru.yandex.direct.validation.constraint.StringConstraints.validUrl
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

private const val DESCRIPTION = "Добавление нового еком домена (или обновление существующего). " +
    "Домен должен присутствовать в выгрузке из самовара: https://yql.yandex-team.ru/Operations/YYHZFAVK8IhSH56gQD5_zIWa0ER31ZuKPBxd3oMLtKI= " +
    "(предварительно нужно получить главное зеркало https://gemini-viewer.in.yandex-team.ru/?url=&action=mirror)"

@Tool(
    name = "Добавление еком домена в базе",
    label = "add_ecom_domain",
    description = DESCRIPTION,
    consumes = AddEcomDomainParam::class,
    type = InternalToolType.WRITER
)
@Action(InternalToolAction.ADD)
@Category(InternalToolCategory.ECOM)
@AccessGroup(InternalToolAccessRole.SUPER)
class EcomDomainAddTool(
    private val ecomDomainsService: EcomDomainsService,
    private val ecomDomainsRepository: EcomDomainsRepository,
    private val geminiClient: GeminiClient
) : BaseInternalTool<AddEcomDomainParam> {

    override fun validate(params: AddEcomDomainParam): ValidationResult<AddEcomDomainParam, Defect<Any>> {
        val vb: ItemValidationBuilder<AddEcomDomainParam, Defect<Any>> = ItemValidationBuilder.of(params)

        vb.item(params.url, "Домен")
            .check(notNull())
            .check(notBlank(), When.isValid())
            .check(validUrl(), When.isValid())

        return vb.result
    }

    override fun process(params: AddEcomDomainParam): InternalToolResult {
        val url = params.url
        var ecomDomain = ecomDomainsService.getEcomDomainsByUrls(listOf(url))[url]

        if (ecomDomain == null) {
            val mainMirror = geminiClient.getMainMirrors(listOf(url))[url]!!
            val urlParts = UrlParts.fromUrl(mainMirror)
            val domain = urlParts.domain
            val schema = urlParts.protocol

            ecomDomain = EcomDomain()
                .withDomain(domain)
                .withSchema(schema)
                .withOffersCount(params.offersCount)
                .withIsPermanent(params.permanent)

            ecomDomainsRepository.insert(listOf(ecomDomain))

            return InternalToolMassResult(listOf(EcomDomainInfo(EcomDomainAction.ADDED, ecomDomain)))
        } else {
            val changes = ModelChanges(ecomDomain.id, EcomDomain::class.java)
                .process(params.permanent, EcomDomain.IS_PERMANENT)
                .applyTo(ecomDomain)

            ecomDomainsRepository.update(listOf(changes))

            return InternalToolMassResult(listOf(EcomDomainInfo(EcomDomainAction.UPDATED, ecomDomain)))
        }
    }

}
