package ru.yandex.direct.internaltools.tools.ess;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.ess.client.EssClient;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolParameter;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.core.input.InternalToolInput;
import ru.yandex.direct.internaltools.core.input.InternalToolInputPreProcessor;
import ru.yandex.direct.internaltools.tools.ess.container.SendEssObjectsOption;
import ru.yandex.direct.internaltools.tools.ess.container.SendLogicObjectsInfo;
import ru.yandex.direct.internaltools.tools.ess.container.SendingObjectsParam;

import static ru.yandex.direct.internaltools.tools.ess.container.SendEssObjectsOption.VIEW_EXAMPLE_OBJECTS;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;

@Tool(
    name = "Отправка объектов в ess",
    label = "sending_objects_to_ess",
    description = "Позволяет отправить объекты без триггеров в базе.\n" +
        "Чтобы посмотреть пример объекта, нужно выбрать опцию \"Посмотреть примеры объектов\".\n" +
        "Если объектов в примере нет, но они ожидаются, значит они не добавлены в выгрузку. Подробнее: " +
        "https://a.yandex-team.ru/arc/trunk/arcadia/direct/apps/event-sourcing-system/common/README.md",
    consumes = SendingObjectsParam.class,
    type = InternalToolType.REPORT
)
@Action(InternalToolAction.ADD)
@Category(InternalToolCategory.ESS)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class SendingObjectsToEssTool extends MassInternalTool<SendingObjectsParam, SendLogicObjectsInfo> {
    private static final Logger logger = LoggerFactory.getLogger(SendingObjectsToEssTool.class);
    private EssClient essClient;

    public SendingObjectsToEssTool(EssClient essClient) {
        this.essClient = essClient;
    }


    @Override
    public List<SendLogicObjectsInfo> getMassData(SendingObjectsParam sendingObjectsParam) {

        var option = SendEssObjectsOption.fromValue(sendingObjectsParam.getOptionName());
        if (VIEW_EXAMPLE_OBJECTS.equals(option)) {
            return List.of(viewExampleObjects(sendingObjectsParam.getLogicProcessorName()));
        }
        return List.of(sendObjectsToEss(sendingObjectsParam));
    }

    private SendLogicObjectsInfo viewExampleObjects(String logicProcessorName) {
        var topicName = essClient.getTopicForProcessor(logicProcessorName);
        return new SendLogicObjectsInfo().withTopic(topicName);
    }

    private SendLogicObjectsInfo sendObjectsToEss(SendingObjectsParam sendingObjectsParam) {
        if (Objects.isNull(sendingObjectsParam.getFile()) && Objects.isNull(sendingObjectsParam.getLogicObjectsToSend())) {
            return new SendLogicObjectsInfo().withStatus("Нужно указать или файл с объектами, или объекты для " +
                "отправки");
        }

        if (Objects.nonNull(sendingObjectsParam.getFile()) && Objects.nonNull(sendingObjectsParam.getLogicObjectsToSend())) {
            return new SendLogicObjectsInfo().withStatus("Нужно указать или файл с объектами, или объекты для " +
                "отправки");
        }

        String objectsToSend = Objects.nonNull(sendingObjectsParam.getFile()) ?
            new String(sendingObjectsParam.getFile()) : sendingObjectsParam.getLogicObjectsToSend();
        try {
            essClient.addLogicObjectsForProcessor(sendingObjectsParam.getShard(),
                sendingObjectsParam.getLogicProcessorName(), objectsToSend, false);
        } catch (RuntimeException e) {
            logger.error("Failed to add logic objects", e);
            return new SendLogicObjectsInfo().withStatus("Не удалось загрузить объекты: " + ExceptionUtils.getStackTrace(e));
        }
        return new SendLogicObjectsInfo().withStatus("OK");
    }

    @Component
    public class SendEssObjectsOptionProcessor implements InternalToolInputPreProcessor<String> {
        @Override
        public <T extends InternalToolParameter> InternalToolInput.Builder<T, String> preSend(
            InternalToolInput.Builder<T, String> inputBuilder) {
            List<String> optionsToChoose = new ArrayList<>();
            List<String> optionNames =
                Arrays.stream(SendEssObjectsOption.values()).map(SendEssObjectsOption::value).collect(Collectors.toUnmodifiableList());
            optionsToChoose.add(null);
            optionsToChoose.addAll(optionNames);
            return inputBuilder
                .withDefaultValue(null)
                .withAllowedValues(optionsToChoose)
                .addValidator(inSet(ImmutableSet.<String>builder().addAll(optionNames).build()));
        }
    }
}
