package ru.yandex.direct.internaltools.tools.ess.blacklist;

import java.util.HashSet;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import org.json.JSONArray;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.essblacklist.EssLogicObjectsBlacklistRepository;
import ru.yandex.direct.core.entity.essblacklist.model.EssBlacklistItem;
import ru.yandex.direct.ess.client.EssClient;
import ru.yandex.direct.ess.common.models.BaseLogicObject;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidFormat;

@Tool(
        name = "Добавление фильтров для объектов в ess-blacklist",
        label = "ess_blacklist_add_tool",
        description = "Добавляя новый фильтр объектов в blacklist мы фактически делаем все объекты, подпадающие" +
                " под фильтр, невидимыми для ess-обработчиков.\n" +
                "Изменения применятся в течении 1 минуты.",
        consumes = EssBlacklistAddParameters.class,
        type = InternalToolType.WRITER
)
@Category(InternalToolCategory.ESS)
@Action(InternalToolAction.ADD)
@AccessGroup({InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPER})
@ParametersAreNonnullByDefault
public class EssBlacklistAddTool extends MassInternalTool<EssBlacklistAddParameters, EssBlacklistItem> {

    private final EssLogicObjectsBlacklistRepository essLogicObjectsRepository;
    private final EssClient essClient;

    @Autowired
    public EssBlacklistAddTool(EssLogicObjectsBlacklistRepository essLogicObjectsRepository, EssClient essClient) {
        this.essLogicObjectsRepository = essLogicObjectsRepository;
        this.essClient = essClient;
    }

    @Override
    public ValidationResult<EssBlacklistAddParameters, Defect> validate(EssBlacklistAddParameters parameters) {
        JsonNode node;

        ItemValidationBuilder<EssBlacklistAddParameters, Defect> validationBuilder =
                ItemValidationBuilder.of(parameters);

        validationBuilder
                .item(parameters.getFilters(), EssBlacklistAddParameters.FILTERS)
                .check(notBlank())
                .check(validJsonListString(parameters.getEssProcessName()));

        return validationBuilder.getResult();
    }

    private Constraint<String, Defect> validJsonListString(String targetProcessorName) {
        var expectedLogicObjectType = essClient.getLogicObjectTypeForProcessor(targetProcessorName);

        return Constraint.fromPredicate(str -> {
            try {
                JavaType listLogicObjectsType = JsonUtils.getTypeFactory().constructCollectionType(List.class,
                        expectedLogicObjectType);
                List<BaseLogicObject> logicObjects = JsonUtils.fromJson("[" + str + "]", listLogicObjectsType);

                return !logicObjects.isEmpty();
            } catch (IllegalArgumentException e) {
                return false;
            }
        }, invalidFormat());
    }

    @Override
    protected List<EssBlacklistItem> getMassData() {
        return essLogicObjectsRepository.getAll();
    }

    @Override
    protected List<EssBlacklistItem> getMassData(EssBlacklistAddParameters parameters) {
        var filters = new HashSet<String>();
        var json = new JSONArray("[" + parameters.getFilters() + "]");
        for (var i = 0; i < json.length(); i++) {
            filters.add(json.getJSONObject(i).toString());
        }

        essLogicObjectsRepository.addItems(parameters.getEssProcessName(), filters,
                parameters.getOperator().getDomainLogin());

        return getMassData();
    }
}
