package ru.yandex.direct.internaltools.tools.ess.mysql2grut

import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.ess.client.EssClient
import ru.yandex.direct.ess.config.mysql2grut.Mysql2GrutReplicationConfig
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup
import ru.yandex.direct.internaltools.core.annotations.tool.Action
import ru.yandex.direct.internaltools.core.annotations.tool.Category
import ru.yandex.direct.internaltools.core.annotations.tool.Tool
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole
import ru.yandex.direct.internaltools.core.enums.InternalToolAction
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory
import ru.yandex.direct.internaltools.core.enums.InternalToolType
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool
import ru.yandex.direct.internaltools.tools.ess.mysql2grut.model.Mysql2GrutResendParam
import ru.yandex.direct.internaltools.tools.ess.mysql2grut.model.Mysql2GrutResendResultRow
import ru.yandex.direct.internaltools.tools.ess.sendcampaign.service.RateLimitingService
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.constraint.StringConstraints
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject

private const val DESCRIPTION =
    """Переотправка кампаний в репликацию в GrUT.
На вход принимает номер кампании.
На данный момент отправляется только сама кампания, ее внешние связи и дочерние объекты не переотправляются!!

Есть ограничение на число объектов ESS, которые можно переотправить в течение ${RateLimitingService.TIME_WINDOW_MINUTES} минут.
Если это ограничение было кем-то превышено - отчёт не запустится.
После успешного выполнения отчёта на экран выведется номер последней обработанной кампании.

Можно управлять через свойства в ppc_properties:
${PpcPropertyNames.MYSQL2GRUT_RESEND_OBJECTS_LIMIT_PROPERTY_NAME} - максимальное число объектов ESS, отправляемое за раз
"""

@Tool(
    name = "Переотправка объектов в репликацию в Grut (бета)",
    label = "mysql2grut_resend_tool",
    description = DESCRIPTION,
    consumes = Mysql2GrutResendParam::class,
    type = InternalToolType.WRITER
)
@Action(InternalToolAction.ADD)
@Category(InternalToolCategory.ESS)
@AccessGroup(
    InternalToolAccessRole.SUPER,
    InternalToolAccessRole.DEVELOPER,
    InternalToolAccessRole.SUPERREADER,
    InternalToolAccessRole.PLACER
)
class Mysql2GrutResendTool(
    private val essClient: EssClient,
    private val shardHelper: ShardHelper,
    ppcPropertiesSupport: PpcPropertiesSupport
) : MassInternalTool<Mysql2GrutResendParam, Mysql2GrutResendResultRow<*>>() {
    private val logicProcessorName = Mysql2GrutReplicationConfig().logicProcessName

    private val stateProperty = ppcPropertiesSupport[PpcPropertyNames.SEND_CAMPAIGNS_TO_ESS_TRANSPORT_STATE]
    private val objectsLimitProperty =
        ppcPropertiesSupport[PpcPropertyNames.SEND_CAMPAIGNS_TO_ESS_TRANSPORT_OBJECTS_LIMIT]

    private val rateLimitingService = RateLimitingService(stateProperty, objectsLimitProperty)

    override fun getMassData(parameter: Mysql2GrutResendParam): List<Mysql2GrutResendResultRow<*>> {
        val currentLimit = rateLimitingService.currentObjectsLimit()
        val allCampaignIds = parameter.parseCampaignsToSend()
        val campaignsToResend = allCampaignIds.take(currentLimit)
        // Пока отправляем только кампании - лимит только на них, когда будут другие объекты, лимит будет на общее число объектов
        sendCampaigns(campaignsToResend)
        val lastSendCampaign = if (allCampaignIds.size > campaignsToResend.size) {
            campaignsToResend.last()
        } else null
        return Mysql2GrutResendResultRow.createResults(lastSendCampaign, campaignsToResend.size)
    }

    private fun sendCampaigns(campaignIds: List<Long>) {
        if (campaignIds.isEmpty()) {
            return
        }
        shardHelper.groupByShard(campaignIds, ShardKey.CID)
            .forEach { shard, data ->
                essClient.addLogicObjectsForProcessor(
                    shard,
                    logicProcessorName,
                    data.map { Mysql2GrutReplicationObject(it) })
                essClient.clearLogicObjects(shard, data.size)
            }
    }

    override fun validate(t: Mysql2GrutResendParam) = validateObject(t) {
        property(t::campaignsToSend) {
            check(StringConstraints.notBlank())
            check {
                val cids = it.trim().split(Regex("\\s+")).filter(String::isNotBlank)
                cids.mapNotNull { cid -> StringConstraints.isPositiveWholeNumber().apply(cid) }
                    .firstOrNull()
            }
        }

        isTimeoutActiveConstraint()
    }

    private fun isTimeoutActiveConstraint() = Constraint<Mysql2GrutResendParam, Defect<*>> { param ->
        rateLimitingService.isTimeoutActive()
    }
}
