package ru.yandex.direct.internaltools.tools.ess.sendcampaign

import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Component
import ru.yandex.direct.common.configuration.CommonConfiguration.DIRECT_EXECUTOR_SERVICE
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.ess.common.models.BaseLogicObject
import ru.yandex.direct.internaltools.tools.ess.sendcampaign.repository.CampaignContentRepository
import java.util.concurrent.ExecutorService

data class LogicObjectBatch(
    val logicProcessName: String,
    val objects: List<BaseLogicObject>,
)

data class CampaignContent(
    val cid: Long,
    val shard: Int,
    val objectBatches: List<LogicObjectBatch>,
) {
    fun countObjects() = objectBatches.sumOf { it.objects.size }
}

@Component
class SendCampaignContentService(
    private val shardHelper: ShardHelper,
    private val repositories: List<CampaignContentRepository<*>>,
    @Qualifier(DIRECT_EXECUTOR_SERVICE) private val executorService: ExecutorService,
) {

    /**
     * Ленивая последовательность объектов для ESS транспорта,
     * сгруппированных по логическому процессору и номеру кампании.
     *
     * Последовательность заканчивается, когда обработаются все кампании
     * или количество объектов ESS превысит лимит
     *
     * @param ignored репозитории с объектами, которые не надо использовать
     * @param limit количество объектов, после превышения которого последовательность заканчивается
     */
    fun getContent(
        cids: Collection<Long>,
        ignored: Set<String> = emptySet(),
        limit: Int? = null,
    ): Sequence<CampaignContent> {
        val allowedRepositories = repositories
            .filter { it.logicProcessName !in ignored }
            .sortedBy { it.logicProcessName }

        val contentSequence = getContentInternal(cids, allowedRepositories)

        return when (limit) {
            null -> contentSequence
            else -> sequence {
                var objectsSent = 0
                for (content in contentSequence) {
                    yield(content)
                    objectsSent += content.countObjects()
                    if (objectsSent > limit) {
                        break
                    }
                }
            }
        }
    }

    /**
     * Ленивая последовательность, разбивающая номера кампаний на чанки
     * и возвращающая для каждой кампании её сгруппированные логические объекты
     */
    private fun getContentInternal(
        cids: Collection<Long>,
        allowedRepositories: List<CampaignContentRepository<*>>
    ): Sequence<CampaignContent> = sequence {
        // Номера кампаний сортируются, удаляются дубликаты
        val cidsChunks = cids
            .toSortedSet()
            .chunked(CAMPAIGNS_CHUNK_SIZE)

        for (cidsChunk in cidsChunks) {
            val contents = getContentInChunk(cidsChunk, allowedRepositories)
            yieldAll(contents)
        }
    }

    private fun getContentInChunk(
        cidsChunk: List<Long>,
        allowedRepositories: List<CampaignContentRepository<*>>,
    ): List<CampaignContent> {
        val cidsByShard: Map<Int, List<Long>> = shardHelper
            .groupByShard(cidsChunk, ShardKey.CID)
            .shardedDataMap

        val objectsByShard = shardHelper.forEachShardParallel({ shard ->
            val shardCids = cidsByShard[shard]
            when {
                shardCids != null -> getContentInShard(shard, shardCids, allowedRepositories)
                else -> emptyList()
            }
        }, executorService)

        return objectsByShard
            .values
            .flatten()
            .sortedBy { it.cid }
    }

    private fun getContentInShard(
        shard: Int,
        cidsChunk: List<Long>,
        allowedRepositories: List<CampaignContentRepository<*>>,
    ): List<CampaignContent> {
        val objectBatchesByCid: Map<Long, List<LogicObjectBatch>> = allowedRepositories
            .asSequence()
            .flatMap { repository -> getObjectsInRepository(shard, cidsChunk, repository) }
            .groupBy({ it.first }) { it.second }

        return cidsChunk.map { cid ->
            val objectBatches = objectBatchesByCid[cid] ?: emptyList()
            CampaignContent(cid, shard, objectBatches)
        }
    }

    private fun <T : BaseLogicObject> getObjectsInRepository(
        shard: Int,
        cidsChunk: List<Long>,
        repository: CampaignContentRepository<T>,
    ): Sequence<Pair<Long, LogicObjectBatch>> {
        return repository
            .getObjectsByCids(shard, cidsChunk)
            .asSequence()
            .map { (cid, objects) ->
                cid to LogicObjectBatch(repository.logicProcessName, objects)
            }
    }

    companion object {
        private const val CAMPAIGNS_CHUNK_SIZE = 50
    }
}
