package ru.yandex.direct.internaltools.tools.ess.sendcampaign

import org.slf4j.LoggerFactory
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.common.db.PpcPropertyNames.SEND_CAMPAIGNS_TO_ESS_TRANSPORT_OBJECTS_LIMIT_NAME
import ru.yandex.direct.ess.client.EssClient
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup
import ru.yandex.direct.internaltools.core.annotations.tool.Action
import ru.yandex.direct.internaltools.core.annotations.tool.Category
import ru.yandex.direct.internaltools.core.annotations.tool.Tool
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole
import ru.yandex.direct.internaltools.core.enums.InternalToolAction
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory
import ru.yandex.direct.internaltools.core.enums.InternalToolType
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool
import ru.yandex.direct.internaltools.tools.ess.sendcampaign.model.SendCampaignParam
import ru.yandex.direct.internaltools.tools.ess.sendcampaign.model.SendCampaignResultRow
import ru.yandex.direct.internaltools.tools.ess.sendcampaign.service.RateLimitingService
import ru.yandex.direct.internaltools.tools.ess.sendcampaign.service.RateLimitingService.Companion.TIME_WINDOW_MINUTES
import ru.yandex.direct.internaltools.tools.ess.sendcampaign.validation.SendCampaignDefects
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.constraint.StringConstraints
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject

private const val DESCRIPTION =
    """Переотправка кампаний, их групп, баннеров, корректировок и т.д. в транспорт на ESS.
На вход принимает номер кампании.
Можно запретить отправку некоторых типов объектов.

Есть ограничение на число объектов ESS, которые можно переотправить в течение $TIME_WINDOW_MINUTES минут.
Если это ограничение было кем-то превышено - отчёт не запустится.
После успешного выполнения отчёта на экран выведется номер последней обработанной кампании.

Можно управлять через свойства в ppc_properties:
$SEND_CAMPAIGNS_TO_ESS_TRANSPORT_OBJECTS_LIMIT_NAME - максимальное число объектов ESS, отправляемое за раз
"""

@Tool(
    name = "Переотправка объектов кампаний в транспорт на ESS (бета)",
    label = "ess_send_campaigns_content",
    description = DESCRIPTION,
    consumes = SendCampaignParam::class,
    type = InternalToolType.WRITER
)
@Action(InternalToolAction.ADD)
@Category(InternalToolCategory.ESS)
@AccessGroup(
    InternalToolAccessRole.SUPER,
    InternalToolAccessRole.DEVELOPER,
    InternalToolAccessRole.SUPERREADER,
    InternalToolAccessRole.PLACER
)
class SendCampaignContentTool(
    private val sendCampaignContentService: SendCampaignContentService,
    private val essClient: EssClient,
    ppcPropertiesSupport: PpcPropertiesSupport
) : MassInternalTool<SendCampaignParam, SendCampaignResultRow<*>>() {

    private val logger = LoggerFactory.getLogger(SendCampaignContentTool::class.java)
    private val stateProperty = ppcPropertiesSupport[PpcPropertyNames.SEND_CAMPAIGNS_TO_ESS_TRANSPORT_STATE]
    private val objectsLimitProperty =
        ppcPropertiesSupport[PpcPropertyNames.SEND_CAMPAIGNS_TO_ESS_TRANSPORT_OBJECTS_LIMIT]

    private val rateLimitingService = RateLimitingService(stateProperty, objectsLimitProperty)

    override fun getMassData(parameter: SendCampaignParam): List<SendCampaignResultRow<*>> {
        val cids = parameter.parseCampaignsToSend()
        val ignoredProcessors = parameter.ignoredProcessors ?: emptySet()
        val processedBatches = process(
            cids,
            ignored = ignoredProcessors,
            sendToEss = parameter.sendingToEss,
            objectsLimit = rateLimitingService.currentObjectsLimit(),
        )

        return createResult(processedBatches)
    }

    private fun process(
        cids: List<Long>,
        ignored: Set<String>,
        sendToEss: Boolean,
        objectsLimit: Int
    ): List<CampaignContent> {
        return when {
            sendToEss -> generateAndSendObjects(cids, ignored, objectsLimit)
            else -> sendCampaignContentService.getContent(cids, ignored).toList()
        }
    }

    private fun generateAndSendObjects(
        cids: List<Long>,
        ignored: Set<String>,
        objectsLimit: Int
    ): List<CampaignContent> {
        val campaignContentSequence =
            sendCampaignContentService.getContent(cids, ignored, limit = objectsLimit)
        val sentBatches = campaignContentSequence
            .mapTo(mutableListOf()) { campaignContent ->
                val (_, shard, objectsBatches) = campaignContent
                for ((logicProcessName, objects) in objectsBatches) {
                    essClient.addLogicObjectsForProcessor(shard, logicProcessName, objects)
                }
                campaignContent
            }

        val objectsSent = sentBatches.sumOf(CampaignContent::countObjects)
        rateLimitingService.updateState(objectsSent)
        logger.info("Sent $objectsSent objects to ESS")
        return sentBatches
    }

    private fun createResult(sentBatches: List<CampaignContent>) =
        SendCampaignResultRow.createResults(
            lastSentCampaign = sentBatches.lastOrNull()?.cid,
            objectsAmounts = calculateInfoByProcessor(sentBatches)
        )

    private fun calculateInfoByProcessor(sentBatches: List<CampaignContent>) =
        sentBatches
            .flatMap { it.objectBatches }
            .groupBy({ it.logicProcessName }) { it.objects.size }
            .mapValues { (_, batchSizes) -> batchSizes.sum() }
            .toSortedMap()

    override fun validate(t: SendCampaignParam) = validateObject(t) {
        property(t::campaignsToSend) {
            check(StringConstraints.notBlank())
            check {
                val cids = it.trim().split(Regex("\\s+")).filter(String::isNotBlank)
                cids.mapNotNull { cid -> StringConstraints.isPositiveWholeNumber().apply(cid) }
                    .firstOrNull()
            }
        }

        isTimeoutActiveConstraint()
    }

    fun isTimeoutActiveConstraint() = Constraint<SendCampaignParam, Defect<*>> {param ->
        if (!param.sendingToEss) {
           return@Constraint null
        }
        rateLimitingService.isTimeoutActive()
    }
}
