package ru.yandex.direct.internaltools.tools.ess.sendcampaign.repository

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.Tables.HIERARCHICAL_MULTIPLIERS
import ru.yandex.direct.dbschema.ppc.enums.HierarchicalMultipliersType
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.ess.config.bsexport.multipliers.BsExportMultipliersConfig
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.BsExportMultipliersObject
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.MultiplierType
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.TimeTargetChangedInfo
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.UpsertInfo
import ru.yandex.direct.tracing.Trace

@Component
class CampaignMultipliersRepository(
    private val dslContextProvider: DslContextProvider,
) : CampaignContentRepository<BsExportMultipliersObject> {

    private val logger = LoggerFactory.getLogger(CampaignMultipliersRepository::class.java)

    override val logicProcessName: String = BsExportMultipliersConfig().logicProcessName

    override fun getObjectsByCids(shard: Int, cids: Collection<Long>): Map<Long, List<BsExportMultipliersObject>> {
        val hierarchical = fetchHierarchicalMultipliers(shard, cids).asSequence()
        val timeTarget = fetchTimeTargetChangedObjects(cids).asSequence()
        return (hierarchical + timeTarget)
            .groupBy({ it.key }) { it.value }
            .mapValues { (_, v) -> v.flatten() }
    }


    private fun fetchHierarchicalMultipliers(shard: Int, cids: Collection<Long>) =
        dslContextProvider.ppc(shard)
            .select(
                HIERARCHICAL_MULTIPLIERS.TYPE,
                HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID,
                HIERARCHICAL_MULTIPLIERS.CID,
            )
            .from(HIERARCHICAL_MULTIPLIERS)
            .innerJoin(CAMPAIGNS).on(CAMPAIGNS.CID.eq(HIERARCHICAL_MULTIPLIERS.CID))
            .where(CAMPAIGNS.CID.`in`(cids))
            .fetchGroups(HIERARCHICAL_MULTIPLIERS.CID) { (dbType, id) ->
                val type = dbType.toBsExportType()
                if (type == null) {
                    logger.warn("Found incorrect hierarchical_multipliers.type field: $dbType")
                }
                type?.let { createUpsertLogicObject(type, id) }
            }
            .filterValues { it != null }

    private fun fetchTimeTargetChangedObjects(cids: Collection<Long>) =
        cids.associateWith { listOf(createTimeTargetChangedObject(it)) }

    companion object {
        private val SUPPORTED_TYPES: Map<String, MultiplierType> = MultiplierType
            .values()
            .flatMap { javaType -> javaType.dbTypes.map { dbType -> dbType.literal to javaType } }
            .toMap()

        private fun HierarchicalMultipliersType.toBsExportType(): MultiplierType? = SUPPORTED_TYPES[this.toString()]

        private fun createUpsertLogicObject(type: MultiplierType, hierarchicalMultiplierId: Long): BsExportMultipliersObject {
            val upsert = UpsertInfo(type, hierarchicalMultiplierId)
            val trace = Trace.current()
            return BsExportMultipliersObject.upsert(upsert, trace.spanId, trace.service, trace.method)
        }

        private fun createTimeTargetChangedObject(cid: Long): BsExportMultipliersObject {
            val info = TimeTargetChangedInfo(cid)
            val trace = Trace.current()
            return BsExportMultipliersObject.timeTargetChanged(info, trace.spanId, trace.service, trace.method)
        }
    }
}
