package ru.yandex.direct.internaltools.tools.ess.sendcampaign.service

import com.fasterxml.jackson.core.JsonParseException
import org.slf4j.LoggerFactory
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.internaltools.tools.ess.sendcampaign.model.SendCampaignState
import ru.yandex.direct.internaltools.tools.ess.sendcampaign.validation.SendCampaignDefects
import ru.yandex.direct.utils.JsonUtils
import ru.yandex.direct.validation.result.Defect
import java.time.Duration
import java.time.Instant

class RateLimitingService(
    private val stateProperty: PpcProperty<String>,
    private val objectsLimitProperty: PpcProperty<Int>,
) {

    fun updateState(newObjectsSent: Int) {
        for (retry in 0..STATE_UPDATE_ATTEMPTS) {
            val oldState = readState()
            val newState = createNewState(oldState, newObjectsSent)
            if (stateProperty.cas(oldState.toJson(), newState.toJson())) {
                return
            }
        }
        logger.warn("Can't compare-and-swap ${stateProperty.name} property")
    }

    private fun createNewState(oldState: SendCampaignState?, newObjectsSent: Int) = when {
        oldState == null || getTimeoutSecondsRemain(oldState) == null -> SendCampaignState(
            objectsSent = newObjectsSent,
            timeWindowStart = Instant.now().epochSecond,
        )
        else -> oldState.copy(objectsSent = oldState.objectsSent + newObjectsSent)
    }

    fun currentObjectsLimit(): Int {
        val objectsLimit = readObjectsLimit()
        val state = readState() ?: return objectsLimit

        return when (getTimeoutSecondsRemain(state)) {
            null -> objectsLimit
            else -> objectsLimit - state.objectsSent
        }
    }

    /**
     * Проверка, можно ли сейчас запускать отчёт
     */
    fun isTimeoutActive(): Defect<*>? {
        val state = readState()
        val objectsLimit = readObjectsLimit()
        val secondsRemain = state?.let { getTimeoutSecondsRemain(it) }

        return when {
            state == null || secondsRemain == null -> null
            state.objectsSent < objectsLimit -> null
            else -> SendCampaignDefects.timeoutIsActive(objectsLimit, secondsRemain)
        }
    }

    /**
     * Сколько секунд отчёт ещё нельзя запускать (или null, если запускать можно)
     */
    private fun getTimeoutSecondsRemain(currentState: SendCampaignState): Long? {
        val timeWindowStart = currentState.timeWindowStart
        val timeWindowEnd = timeWindowStart + TIME_WINDOW_DURATION.seconds
        val now = Instant.now().epochSecond

        return when {
            timeWindowEnd < now -> null
            else -> timeWindowEnd - now
        }
    }

    private fun readObjectsLimit(): Int = objectsLimitProperty.get() ?: 10000

    private fun readState(): SendCampaignState? {
        val json = stateProperty.get() ?: return null
        return try {
            JsonUtils.fromJson(json, SendCampaignState::class.java)
        } catch (ex: JsonParseException) {
            logger.warn(ex.message)
            null
        }
    }

    companion object {
        const val TIME_WINDOW_MINUTES = 10L
        private val TIME_WINDOW_DURATION = Duration.ofMinutes(TIME_WINDOW_MINUTES)
        private const val STATE_UPDATE_ATTEMPTS = 10
        private val logger = LoggerFactory.getLogger(RateLimitingService::class.java)
    }
}

private fun <T> T?.toJson(): String? = this?.let { JsonUtils.toDeterministicJson(it) }
