package ru.yandex.direct.internaltools.tools.essmoderation;

import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.type.TypeReference;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.moderation.model.ObjectWithEnabledEssModeration;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.essmoderation.model.EnableEssModerationParameter;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.common.db.PpcPropertyNames.LIST_OF_OBJECTS_WITH_ENABLED_ESS_MODERATION;
import static ru.yandex.direct.core.entity.moderation.model.ObjectWithEnabledEssModeration.BANNER_ID;
import static ru.yandex.direct.core.entity.moderation.model.ObjectWithEnabledEssModeration.CAMPAIGN_ID;
import static ru.yandex.direct.internaltools.tools.essmoderation.model.EnableEssModerationParameter.OBJECT_ID_PROPERTY;
import static ru.yandex.direct.internaltools.tools.essmoderation.model.EnableEssModerationParameter.OBJECT_TYPE_PROPERTY;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Tool(
        name = "Список объектов с включенной ess модерацией",
        label = "enable_ess_moderation",
        description = "Редактирование списка объектов для которых надо включить ess модерацию",
        consumes = EnableEssModerationParameter.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.MODERATE)
@AccessGroup({InternalToolAccessRole.INTERNAL_USER})
@ParametersAreNonnullByDefault
public class EnableEssModerationTool extends MassInternalTool<EnableEssModerationParameter,
        ObjectWithEnabledEssModeration> {

    private static final TypeReference<List<ObjectWithEnabledEssModeration>> TYPE_REFERENCE_OBJECTS_WITH_ENABLED_ESS_MODERATION = new TypeReference<>() {
    };

    private final PpcPropertiesSupport ppcPropertiesSupport;

    @Autowired
    public EnableEssModerationTool(PpcPropertiesSupport ppcPropertiesSupport) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    @Override
    public ValidationResult<EnableEssModerationParameter, Defect> validate(EnableEssModerationParameter params) {
        ItemValidationBuilder<EnableEssModerationParameter, Defect> vb =
                ItemValidationBuilder.of(params, Defect.class);
        vb.item(params.getObjectId(), OBJECT_ID_PROPERTY)
                .check(notNull());
        vb.item(params.getObjectType(), OBJECT_TYPE_PROPERTY)
                .check(notNull());
        return vb.getResult();
    }

    @Override
    protected List<ObjectWithEnabledEssModeration> getMassData() {
        var objectsWithEnabledModerationJson =
                ppcPropertiesSupport.get(LIST_OF_OBJECTS_WITH_ENABLED_ESS_MODERATION).getOrDefault("[]");
        return JsonUtils.fromJson(objectsWithEnabledModerationJson, TYPE_REFERENCE_OBJECTS_WITH_ENABLED_ESS_MODERATION);
    }

    @Override
    protected List<ObjectWithEnabledEssModeration> getMassData(EnableEssModerationParameter parameter) {
        var updatedData = parameter.getToRemove()
                ? disableEssModeration(getMassData(), parameter)
                : enableEssModeration(getMassData(), parameter);

        ppcPropertiesSupport.get(LIST_OF_OBJECTS_WITH_ENABLED_ESS_MODERATION)
                .set(JsonUtils.toJson(updatedData));

        return updatedData;
    }

    private List<ObjectWithEnabledEssModeration> disableEssModeration(List<ObjectWithEnabledEssModeration> data,
                                                                      EnableEssModerationParameter parameter) {
        final ModelProperty<ObjectWithEnabledEssModeration, Long> propertyForObjectId =
                getPropertyForObjectId(parameter);

        return data.stream()
                .filter(obj -> !parameter.getObjectId().equals(propertyForObjectId.get(obj)))
                .collect(toList());
    }

    private List<ObjectWithEnabledEssModeration> enableEssModeration(List<ObjectWithEnabledEssModeration> data,
                                                                     EnableEssModerationParameter parameter) {
        final ModelProperty<ObjectWithEnabledEssModeration, Long> propertyForObjectId =
                getPropertyForObjectId(parameter);

        boolean alreadyExists = data.stream()
                .anyMatch(obj -> parameter.getObjectId().equals(propertyForObjectId.get(obj)));
        if (alreadyExists) {
            return data;
        }

        var objectWithEnabledEssModerationToAdd = new ObjectWithEnabledEssModeration()
                .withLastChange(LocalDateTime.now());
        propertyForObjectId.set(objectWithEnabledEssModerationToAdd, parameter.getObjectId());
        data.add(objectWithEnabledEssModerationToAdd);

        return data;
    }

    private ModelProperty<ObjectWithEnabledEssModeration, Long> getPropertyForObjectId(
            EnableEssModerationParameter parameter) {
        switch (parameter.getObjectType()) {
            case CAMPAIGN:
                return CAMPAIGN_ID;
            case BANNER:
                return BANNER_ID;
            default:
                throw new IllegalArgumentException("Object type is not supported");
        }
    }
}
