package ru.yandex.direct.internaltools.tools.essmoderation.beta;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.type.TypeReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.log.container.ModerationLogEntry;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.common.log.LogHelper.DATE_TIME_FORMATTER;

@ParametersAreNonnullByDefault
class LogFilesReader {
    private static final Logger logger = LoggerFactory.getLogger(LogFilesReader.class);
    private static final TypeReference<ModerationLogEntry<ModerationLogEntryData>> TYPE_REF = new TypeReference<>() {};

    private final Path appPath;
    private final String logType;

    public LogFilesReader(Path appPath, String logType) {
        this.appPath = appPath;
        this.logType = logType;
    }

    public List<ModerationLogEntry<ModerationLogEntryData>> readEntries(LocalDateTime startTime) throws IOException {
        logger.debug("Trying to find '{}' logs in '{}'", logType, appPath.toAbsolutePath());

        Path logsDirPath = appPath.resolve("logs");
        String startLogFile = logType + ".log." + DateTimeFormatter.ofPattern("yyyyMMdd").format(startTime);
        String[] actualFiles = logsDirPath.toFile().list(
                (dir, name) -> name.startsWith(logType + ".log") && startLogFile.compareToIgnoreCase(name) <= 0
        );

        logger.info("Found ess_moderation logs: {}", actualFiles);

        List<ModerationLogEntry<ModerationLogEntryData>> entries = new ArrayList<>();
        for (var logFile : actualFiles) {
            entries.addAll(readLogFile(logsDirPath.resolve(logFile), startTime));
        }
        entries.sort(Comparator.comparing(LogFilesReader::toLocalDateTime));
        return entries;
    }

    public static List<ModerationLogEntry<ModerationLogEntryData>> readLogFile(Path logFile, LocalDateTime startTime)
            throws IOException {

        try (Stream<String> lines = Files.lines(logFile)) {
            return lines.map(line -> JsonUtils.fromJson(line, TYPE_REF))
                    .filter(logEntry -> toLocalDateTime(logEntry).isAfter(startTime))
                    .collect(toList());
        }
    }

    private static LocalDateTime toLocalDateTime(ModerationLogEntry<ModerationLogEntryData> logEntry) {
        return LocalDateTime.parse(logEntry.getLogTime(), DATE_TIME_FORMATTER);
    }

}
