package ru.yandex.direct.internaltools.tools.feature.container;

import java.util.List;

import one.util.streamex.StreamEx;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.feature.container.ChiefRepresentativeWithClientFeature;
import ru.yandex.direct.core.entity.feature.container.FeatureTextIdToClientIdState;
import ru.yandex.direct.core.entity.feature.container.FeatureTextIdToLoginState;
import ru.yandex.direct.core.entity.feature.container.FeatureTextIdToPercent;
import ru.yandex.direct.core.entity.feature.container.FeatureTextIdToRole;
import ru.yandex.direct.core.entity.feature.container.LoginClientIdChiefLoginWithFeature;
import ru.yandex.direct.core.entity.feature.container.LoginClientIdChiefLoginWithState;
import ru.yandex.direct.core.entity.feature.model.Feature;
import ru.yandex.direct.core.entity.feature.model.FeatureState;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.internaltools.utils.ToolParameterUtils;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.internaltools.tools.feature.InternalToolsConstants.INTERNAL;
import static ru.yandex.direct.internaltools.tools.feature.InternalToolsConstants.INTERNAL_ROLES;
import static ru.yandex.direct.internaltools.tools.feature.container.InternalToolsViewOrDeleteFeatureInfo.TRUE_VALUE;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

public class InternalToolsFeatureConverter {
    public static FeatureTextIdToPercent toFeatureToPercent(InternalToolsFeaturePercent internalToolsFeaturePercent) {
        return new FeatureTextIdToPercent()
                .withPercent(internalToolsFeaturePercent.getPercent().intValue())
                .withTextId(internalToolsFeaturePercent.getTextId());
    }

    public static InternalToolsFeaturePercentResponse toInternalToolsFeaturePercentResponse(
            InternalToolsFeaturePercent internalToolsFeaturePercent) {
        return new InternalToolsFeaturePercentResponse()
                .withPercent(internalToolsFeaturePercent.getPercent().intValue())
                .withTextId(internalToolsFeaturePercent.getTextId());
    }

    public static InternalToolsFeature toInternalToolsFeatureResponse(Feature feature) {
        return new InternalToolsFeature()
                .withId(feature.getId())
                .withFeatureTextId(feature.getFeatureTextId())
                .withFeaturePublicName(feature.getFeaturePublicName())
                .withFeatureTicket(feature.getFeatureTicket())
                .withIsAccessibleAfterDisabling(feature.getSettings().getIsAccessibleAfterDisabling())
                .withRoles(feature.getSettings().getRoles())
                .withPercent(feature.getSettings().getPercent())
                .withCanEnable(feature.getSettings().getCanEnable())
                .withCanDisable(feature.getSettings().getCanDisable())
                .withIsPublic(feature.getSettings().getIsPublic())
                .withFeatureType(
                        ifNotNull(FeatureName.fromString(feature.getFeatureTextId()), FeatureName::getFeatureType));
    }

    public static InternalToolsViewOrDeleteFeatureInfo toInternalToolsViewOrDeleteFeatureInfo(Feature existingFeature) {
        return new InternalToolsViewOrDeleteFeatureInfo()
                .withFeatureTextId(existingFeature.getFeatureTextId())
                .withFeaturePublicName(existingFeature.getFeaturePublicName())
                .withFeatureTicket(existingFeature.getFeatureTicket())
                .withId(existingFeature.getId())
                .withIsTurnOn(isTurnOn(existingFeature.getSettings().getPercent()))
                .withIsPublic(isPublic(existingFeature.getSettings().getIsPublic()));
    }

    public static InternalToolsAddFeatureInfo toInternalToolsAddFeatureInfo(Feature existingFeature) {
        return new InternalToolsAddFeatureInfo()
                .withFeatureTextId(existingFeature.getFeatureTextId())
                .withFeaturePublicName(existingFeature.getFeaturePublicName())
                .withFeatureTicket(existingFeature.getFeatureTicket())
                .withFeatureId(existingFeature.getId())
                .withIsTurnOn(isTurnOn(existingFeature.getSettings().getPercent()))
                .withIsPublic(isPublic(existingFeature.getSettings().getIsPublic()))
                .withOriginalOwner(existingFeature.getSettings().getOriginalOwner())
                .withOriginalWatchers(JsonUtils.toJsonNullable(existingFeature.getSettings().getOriginalWatchers()));
    }

    private static String isTurnOn(Integer percent) {
        return percent == 100 ? TRUE_VALUE : null;
    }

    private static String isPublic(Boolean isPublic) {
        return isPublic != null && isPublic ? TRUE_VALUE : null;
    }

    public static InternalToolsViewOrDeleteFeatureInfo toInternalToolsViewOrDeleteFeatureInfo(TranslationService translationService,
                                                                                              FeatureName featureName) {
        return new InternalToolsViewOrDeleteFeatureInfo()
                .withFeatureTextId(featureName.getName())
                .withFeaturePublicName(translationService.translate(featureName.getHumanReadableName()));
    }

    public static List<FeatureTextIdToLoginState> toFeatureTextIdToLoginStateList(
            InternalToolsFeatureIdClientLoginsState internalToolsFeatureIdClientLoginsState) {
        List<String> clientLogins = parseLogins(internalToolsFeatureIdClientLoginsState);
        List<String> featureTextIds = parseFeatures(internalToolsFeatureIdClientLoginsState);
        FeatureState featureState = toFeatureState(internalToolsFeatureIdClientLoginsState.getState());

        return StreamEx.of(clientLogins)
                .map(String::trim)
                .cross(featureTextIds)
                .mapKeyValue(
                        (clientLogin, featureTextId) -> new FeatureTextIdToLoginState()
                                .withTextId(featureTextId)
                                .withState(featureState)
                                .withLogin(clientLogin)
                ).toList();
    }

    public static List<String> parseLogins(InternalToolsFeatureIdClientLoginsState parameter) {
        return List.copyOf(ToolParameterUtils.parseLogins(parameter.getClientLogins()));
    }

    public static List<String> parseFeatures(InternalToolsFeatureIdClientLoginsState parameter) {
        return StreamEx.split(parameter.getFeatureTextId(), ',').map(String::trim).toList();
    }

    public static List<FeatureTextIdToClientIdState> toFeatureTextIdToClientIdStateList(
            InternalToolsFeatureIdClientIdsState internalToolsFeatureIdClientIdsState) {
        List<ClientId> clientIds = parseClientIds(internalToolsFeatureIdClientIdsState);
        List<String> featureTextIds = parseFeatures(internalToolsFeatureIdClientIdsState);
        FeatureState featureState = toFeatureState(internalToolsFeatureIdClientIdsState.getState());
        return StreamEx.of(clientIds).cross(featureTextIds).mapKeyValue(
                (clientId, featureTextId) -> new FeatureTextIdToClientIdState()
                        .withTextId(featureTextId)
                        .withState(featureState)
                        .withClientId(clientId)
        ).toList();
    }

    public static List<ClientId> parseClientIds(InternalToolsFeatureIdClientIdsState parameter) {
        return StreamEx.split(parameter.getClientIds(), ',')
                .map(String::trim)
                .map(Long::valueOf)
                .map(ClientId::fromLong)
                .toList();
    }

    public static List<String> parseFeatures(InternalToolsFeatureIdClientIdsState parameter) {
        return StreamEx.split(parameter.getFeatureTextId(), ',').map(String::trim).toList();
    }

    public static InternalToolsLoginWithChiefLoginResponse toInternalToolsLoginsWithChiefLoginsResponse(
            LoginClientIdChiefLoginWithFeature loginClientIdChiefLoginWithState,
            boolean reset
    ) {
        return new InternalToolsLoginWithChiefLoginResponse()
                .withClientId(loginClientIdChiefLoginWithState.getClientId())
                .withLogin(loginClientIdChiefLoginWithState.getLogin())
                .withIsPresent(!reset)
                .withIsEnabled(fromFeatureState(loginClientIdChiefLoginWithState.getFeatureState()))
                .withChiefLogin(loginClientIdChiefLoginWithState.getChiefLogin())
                .withTextId(loginClientIdChiefLoginWithState.getFeatureTextId());
    }

    public static InternalToolsLoginResponse toInternalToolsLoginResponse(
            LoginClientIdChiefLoginWithState loginClientIdChiefLoginWithState,
            boolean reset
    ) {
        return new InternalToolsLoginResponse()
                .withClientId(loginClientIdChiefLoginWithState.getClientId())
                .withLogin(loginClientIdChiefLoginWithState.getChiefLogin())
                .withIsPresent(!reset)
                .withIsEnabled(fromFeatureState(loginClientIdChiefLoginWithState.getFeatureState()));
    }

    public static InternalToolsClientFeature toInternalToolsClientFeature(
            ChiefRepresentativeWithClientFeature chiefRepresentativeWithClientFeature) {
        return new InternalToolsClientFeature()
                .withClientId(chiefRepresentativeWithClientFeature.getClientFeature().getClientId().asLong())
                .withIsEnabled(fromFeatureState(chiefRepresentativeWithClientFeature.getClientFeature().getState()))
                .withChiefUid(chiefRepresentativeWithClientFeature.getChiefUid())
                .withChiefLogin(chiefRepresentativeWithClientFeature.getChiefLogin());
    }

    public static List<FeatureTextIdToRole> toFeatureToRoleList(
            InternalToolsFeatureRolesState internalToolsFeatureRolesState) {
        List<String> roles = StreamEx.split(internalToolsFeatureRolesState.getRoles(), ',')
                .map(String::trim)
                .map(String::toUpperCase)
                .toList();
        if (roles.contains(INTERNAL)) {
            roles.remove(INTERNAL);
            roles.addAll(INTERNAL_ROLES);
        }
        return StreamEx.of(roles).map(role -> new FeatureTextIdToRole()
                .withTextId(internalToolsFeatureRolesState.getTextId())
                .withRoleName(role))
                .toList();

    }

    public static InternalToolsFeatureRole toInternalToolsFeatureRole(FeatureTextIdToRole featureTextIdToRole,
                                                                      Boolean state) {
        return new InternalToolsFeatureRole()
                .withTextId(featureTextIdToRole.getTextId())
                .withRole(featureTextIdToRole.getRoleName())
                .withState(state);
    }

    public static InternalToolsFeaturePublicity toInternalToolsFeaturePublicity(String textId, boolean isPublic) {
        return new InternalToolsFeaturePublicity()
                .withTextId(textId)
                .withIsPublic(isPublic);
    }

    public static Boolean fromFeatureState(FeatureState featureState) {
        return FeatureState.ENABLED.equals(featureState);
    }

    public static FeatureState toFeatureState(Boolean featureState) {
        return featureState ? FeatureState.ENABLED : FeatureState.DISABLED;
    }
}
