package ru.yandex.direct.internaltools.tools.feature.tool;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.core.entity.abt.repository.UaasInfoRepository;
import ru.yandex.direct.core.entity.abt.service.EnvironmentNameGetter;
import ru.yandex.direct.core.entity.abt.service.UaasConditionEvaluator;
import ru.yandex.direct.core.entity.abt.service.UaasConditionParams;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.InternalToolProxy;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.feature.container.AbConditionsCheckParam;

@Tool(
        name = "Пороверка condition для ab-экспериментов для клиента",
        label = "ab_conditions_check",
        description = "Пороверка condition для ab-экспериментов для клиента",
        consumes = AbConditionsCheckParam.class,
        type = InternalToolType.REPORT
)
@Action(InternalToolAction.SHOW)
@Category(InternalToolCategory.FEATURES)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.SUPERREADER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class AbConditionsCheckTool implements BaseInternalTool<AbConditionsCheckParam> {
    @Autowired
    private UaasInfoRepository uaasInfoRepository;
    @Autowired
    private UaasConditionEvaluator uaasConditionEvaluator;
    @Autowired
    private ShardHelper shardHelper;
    @Autowired
    private EnvironmentNameGetter environmentNameGetter;

    @Autowired
    private UserService userService;

    @Override
    public InternalToolResult process(AbConditionsCheckParam parameter) {
        if (Objects.isNull(parameter.getClientId()) && (Objects.isNull(parameter.getLogin()) || "".equals(parameter.getLogin()))) {
            return new InternalToolResult().withMessage("Не указаны параметры clientId или login");
        }

        var clientId = parameter.getClientId() != null ? parameter.getClientId() :
                userService.getClientIdByLogin(parameter.getLogin());

        if (Objects.isNull(clientId)) {
            return new InternalToolResult().withMessage("Клиента " + parameter.getLogin() + " не существует");
        }
        Map<Long, Boolean> existingClients = shardHelper.getExistingClientIds(List.of(clientId));
        if (!existingClients.get(clientId)) {
            return new InternalToolResult().withMessage("Клиент с clientId " + clientId + " не " +
                    "существует");
        }
        var localeOptional = HttpUtil.getCurrentLocale();
        var uaasConditionParams = uaasInfoRepository.getUaasConditionsParams(List.of(clientId));
        var paramsForClientId = uaasConditionParams.get(clientId);
        paramsForClientId.withInterfaceLang(localeOptional.map(Locale::getLanguage).orElse(null));
        paramsForClientId.withEnv(environmentNameGetter.get());
        try {
            var isSuitable = uaasConditionEvaluator.evaluateUnsafe(parameter.getCondition(),
                    uaasConditionParams.get(clientId));
            return new InternalToolResult().withMessage(String.valueOf(isSuitable));
        } catch (RuntimeException e) {
            return new InternalToolResult().withMessage(e.getMessage());
        }
    }

    @Override
    public InternalToolProxy.Builder<AbConditionsCheckParam> preCreate(InternalToolProxy.Builder<AbConditionsCheckParam> builder) {
        var availableFields =
                Arrays.stream(UaasConditionParams.class.getDeclaredFields()).map(Field::getName).collect(Collectors.joining(", "));
        List<String> disclaimers = new ArrayList<>(builder.getDisclaimers());
        disclaimers.add("Доступные поля для conditon: " + availableFields);
        return builder.withDisclaimers(disclaimers);
    }
}
