package ru.yandex.direct.internaltools.tools.feature.tool;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.feature.container.LoginClientIdChiefLoginWithState;
import ru.yandex.direct.core.entity.feature.service.FeatureManagingService;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeatureIdClientIdsState;
import ru.yandex.direct.internaltools.tools.feature.container.InternalToolsLoginResponse;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.internaltools.tools.feature.InternalToolsConstants.NUMBERS_SEPARATED_BY_COMMAS_PATTERN;
import static ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeatureConverter.parseClientIds;
import static ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeatureConverter.parseFeatures;
import static ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeatureConverter.toFeatureTextIdToClientIdStateList;
import static ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeatureConverter.toInternalToolsLoginResponse;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.matchPattern;

@Tool(
        name = "Управление доступом к фичам клиентов по clientId",
        label = "switch_feature_for_client_by_client_id",
        description =
                "Включение и выключение фичи клиентам по списку ClientID. Если действие успешно, на этой странице появится таблица с данными клиента. Доступ появится у всех представителей клиента. \n"
                        + "Для удаления клиента (выключения доступа к фиче) – отключите чекбокс «включить фичу».\n"
                        + "Для сброса фичи (удаления записи о доступе из таблицы) - включите чекбокс «сбросить фичу».",
        consumes = InternalToolsFeatureIdClientIdsState.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.SET)
@Category(InternalToolCategory.FEATURES)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class FeatureClientsSwitchByClientIdTool extends MassInternalTool<InternalToolsFeatureIdClientIdsState, InternalToolsLoginResponse> {
    @Autowired
    private FeatureManagingService clientFeatureService;

    @Override
    protected List<InternalToolsLoginResponse> getMassData(
            InternalToolsFeatureIdClientIdsState parameter) {
        Result<List<LoginClientIdChiefLoginWithState>> operationResult;
        var reset = parameter.getReset() != null ? parameter.getReset() : false;
        if (reset) {
            operationResult = clientFeatureService.deleteFeaturesFromClientIds(
                    parseFeatures(parameter), parseClientIds(parameter)
            );
        } else {
            operationResult = clientFeatureService.switchFeaturesStateForClientIds(
                    toFeatureTextIdToClientIdStateList(parameter)
            );
        }

        if (hasValidationIssues(operationResult)) {
            throw new InternalToolValidationException("").withValidationResult(operationResult.getValidationResult());
        }
        return mapList(operationResult.getResult(), result ->
                toInternalToolsLoginResponse(result, reset));
    }

    @Override
    public ValidationResult<InternalToolsFeatureIdClientIdsState, Defect> validate(
            InternalToolsFeatureIdClientIdsState internalToolsFeatureIdClientIdsState) {
        ItemValidationBuilder<InternalToolsFeatureIdClientIdsState, Defect> validationBuilder =
                ItemValidationBuilder.of(internalToolsFeatureIdClientIdsState);
        validationBuilder.check(notNull());
        validationBuilder.item(internalToolsFeatureIdClientIdsState.getFeatureTextId(),
                InternalToolsFeatureIdClientIdsState.ID).check(notNull());
        validationBuilder.item(internalToolsFeatureIdClientIdsState.getClientIds().replaceAll("\\s+", ""),
                InternalToolsFeatureIdClientIdsState.CLIENT_IDS)
                .check(notNull())
                .check(matchPattern(NUMBERS_SEPARATED_BY_COMMAS_PATTERN), When.notNull());

        return validationBuilder.getResult();
    }
}
