package ru.yandex.direct.internaltools.tools.feature.tool;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.feature.container.LoginClientIdChiefLoginWithFeature;
import ru.yandex.direct.core.entity.feature.service.FeatureManagingService;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeatureIdClientLoginsState;
import ru.yandex.direct.internaltools.tools.feature.container.InternalToolsLoginWithChiefLoginResponse;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.internaltools.tools.feature.InternalToolsConstants.LOGINS_SEPARATED_BY_COMMAS_PATTERN;
import static ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeatureConverter.parseFeatures;
import static ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeatureConverter.parseLogins;
import static ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeatureConverter.toFeatureTextIdToLoginStateList;
import static ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeatureConverter.toInternalToolsLoginsWithChiefLoginsResponse;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.matchPattern;

@Tool(
        name = "Управление доступом к фичам клиентов по логину",
        label = "switch_feature_for_client_by_login",
        description =
                "Включение и выключение фичи клиентам по списку логинов (в базе опция добавляется на ClientID, доступ будет у всех представителей). Если действие успешно, на этой странице появится таблица с данными клиента. \n"
                        + "Для удаления клиента (выключения доступа к фиче) – отключите чекбокс «включить фичу».\n"
                        + "Для сброса фичи (удаления записи о доступе из таблицы) - включите чекбокс «сбросить фичу».",
        consumes = InternalToolsFeatureIdClientLoginsState.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.SET)
@Category(InternalToolCategory.FEATURES)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPPORT})
@ParametersAreNonnullByDefault
public class FeatureClientsSwitchByLoginsTool extends MassInternalTool<InternalToolsFeatureIdClientLoginsState, InternalToolsLoginWithChiefLoginResponse> {
    @Autowired
    private FeatureManagingService clientFeatureService;


    @Override
    protected List<InternalToolsLoginWithChiefLoginResponse> getMassData(
            InternalToolsFeatureIdClientLoginsState parameter) {
        Result<List<LoginClientIdChiefLoginWithFeature>> operationResult;
        var reset = parameter.getReset() != null ? parameter.getReset() : false;
        if (reset) {
            operationResult = clientFeatureService.deleteFeaturesFromClientIdsByLogins(
                    parseFeatures(parameter), parseLogins(parameter)
            );
        } else {
            operationResult = clientFeatureService.switchFeaturesStateForClientIdsByLogins(
                    toFeatureTextIdToLoginStateList(parameter)
            );
        }
        if (hasValidationIssues(operationResult)) {
            throw new InternalToolValidationException("").withValidationResult(operationResult.getValidationResult());
        }
        return mapList(operationResult.getResult(), result ->
                toInternalToolsLoginsWithChiefLoginsResponse(result, reset));
    }

    @Override
    public ValidationResult<InternalToolsFeatureIdClientLoginsState, Defect> validate(
            InternalToolsFeatureIdClientLoginsState internalToolsFeatureIdClientLoginsState) {
        ItemValidationBuilder<InternalToolsFeatureIdClientLoginsState, Defect> validationBuilder =
                ItemValidationBuilder.of(internalToolsFeatureIdClientLoginsState);
        validationBuilder.check(notNull());
        validationBuilder.item(internalToolsFeatureIdClientLoginsState.getFeatureTextId(),
                InternalToolsFeatureIdClientLoginsState.ID).check(notNull());
        validationBuilder.item(internalToolsFeatureIdClientLoginsState.getClientLogins().replaceAll("\\s+", ""),
                InternalToolsFeatureIdClientLoginsState.LOGINS)
                .check(notNull())
                .check(matchPattern(LOGINS_SEPARATED_BY_COMMAS_PATTERN), When.isValidAnd(When.notNull()));

        return validationBuilder.getResult();
    }
}
