package ru.yandex.direct.internaltools.tools.feature.tool;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.feature.service.FeatureManagingService;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolMassResult;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeaturePublicityState;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeatureConverter.toInternalToolsFeaturePublicity;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Tool(
        name = "Включение/выключение публичности фичи",
        label = "update_feature_publicity",
        description = "Редактировать публичность фичи. \n" +
                "Если фича помечена как публичная, то она будет показана клиенту в интерфейсе.",
        consumes = InternalToolsFeaturePublicityState.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.SET)
@Category(InternalToolCategory.FEATURES)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.MANAGER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class FeaturePublicityUpdateTool implements BaseInternalTool<InternalToolsFeaturePublicityState> {

    @Autowired
    private FeatureManagingService featureManagingService;

    @Override
    public ValidationResult<InternalToolsFeaturePublicityState, Defect> validate(
            InternalToolsFeaturePublicityState publicityState) {
        ItemValidationBuilder<InternalToolsFeaturePublicityState, Defect> vb = ItemValidationBuilder.of(publicityState);
        vb.item(publicityState.getTextId(), InternalToolsFeaturePublicityState.FEATURE_TEXT_ID)
                .check(notNull())
                .check(notBlank());
        vb.item(publicityState.getAction(), InternalToolsFeaturePublicityState.PUBLICITY_ACTION)
                .check(notNull());
        return vb.getResult();
    }

    @Override
    public InternalToolResult process(InternalToolsFeaturePublicityState publicityState) {
        var featureTextId = publicityState.getTextId();
        var isPublic = publicityState.getAction() == InternalToolsFeaturePublicityState.Action.ENABLE_PUBLICITY;
        Result<Long> result = featureManagingService.updateFeaturePublicity(featureTextId, isPublic);
        if (hasValidationIssues(result)) {
            throw new InternalToolValidationException("").withValidationResult(result.getValidationResult());
        }
        var internalToolsFeaturePublicity = toInternalToolsFeaturePublicity(featureTextId, isPublic);
        return new InternalToolMassResult<>(List.of(internalToolsFeaturePublicity));
    }
}
