package ru.yandex.direct.internaltools.tools.feature.tool;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.feature.container.FeatureTextIdToRole;
import ru.yandex.direct.core.entity.feature.service.FeatureManagingService;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolMassResult;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeatureRolesState;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.internaltools.tools.feature.InternalToolsConstants.LOGINS_SEPARATED_BY_COMMAS_PATTERN;
import static ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeatureConverter.toFeatureToRoleList;
import static ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeatureConverter.toInternalToolsFeatureRole;
import static ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeatureRolesState.ROLES;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.matchPattern;

@Tool(
        name = "Обновление ролей для фичи",
        label = "update_feature_roles",
        description =
                "Открытие доступа к фиче для внутренних ролей. Добавьте или удалите роль при необходимости. Доступные роли: \n"
                        + "SUPER\n"
                        + "SUPERREADER\n"
                        + "SUPPORT\n"
                        + "PLACER\n"
                        + "MANAGER\n"
                        + "AGENCY\n"
                        + "INTERNAL",
        consumes = InternalToolsFeatureRolesState.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.SET)
@Category(InternalToolCategory.FEATURES)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class FeatureRolesUpdateTool implements BaseInternalTool<InternalToolsFeatureRolesState> {

    @Autowired
    private FeatureManagingService featureManagingService;

    @Override
    public ValidationResult<InternalToolsFeatureRolesState, Defect> validate(
            InternalToolsFeatureRolesState internalToolsFeatureRolesState) {
        ItemValidationBuilder<InternalToolsFeatureRolesState, Defect> validationBuilder =
                ItemValidationBuilder.of(internalToolsFeatureRolesState);
        validationBuilder.check(notNull());
        validationBuilder
                .item(internalToolsFeatureRolesState.getTextId(), InternalToolsFeatureRolesState.FEATURE_TEXT_ID)
                .check(notNull());
        validationBuilder.item(internalToolsFeatureRolesState.getRoles().replaceAll("\\s+", ""), ROLES)
                .check(notNull())
                .check(matchPattern(LOGINS_SEPARATED_BY_COMMAS_PATTERN), When.notNull());

        return validationBuilder.getResult();
    }

    @Override
    public InternalToolResult process(InternalToolsFeatureRolesState internalToolsFeatureRolesState) {
        List<FeatureTextIdToRole> featuresToRoles =
                toFeatureToRoleList(internalToolsFeatureRolesState);
        Result<List<Long>> operationResult =
                featureManagingService.updateFeaturesRoles(featuresToRoles, internalToolsFeatureRolesState.getState());
        if (hasValidationIssues(operationResult)) {
            throw new InternalToolValidationException("").withValidationResult(operationResult.getValidationResult());
        }
        return new InternalToolMassResult<>(StreamEx.of(featuresToRoles)
                .distinct()
                .map(featureToRole -> toInternalToolsFeatureRole(featureToRole,
                        internalToolsFeatureRolesState.getState()))
                .toList());

    }


}
