package ru.yandex.direct.internaltools.tools.feature.tool

import one.util.streamex.StreamEx
import ru.yandex.direct.core.entity.feature.service.FeatureCache
import ru.yandex.direct.core.entity.feature.service.FeatureManagingService
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.validation.ValidationUtils
import ru.yandex.direct.internaltools.core.BaseInternalTool
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup
import ru.yandex.direct.internaltools.core.annotations.tool.Action
import ru.yandex.direct.internaltools.core.annotations.tool.Category
import ru.yandex.direct.internaltools.core.annotations.tool.Tool
import ru.yandex.direct.internaltools.core.container.InternalToolResult
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole
import ru.yandex.direct.internaltools.core.enums.InternalToolAction
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory
import ru.yandex.direct.internaltools.core.enums.InternalToolType
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException
import ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeatureUpdateWebEditableParam
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import java.util.regex.Pattern
import javax.annotation.ParametersAreNonnullByDefault

@Tool(
    name = "Обновление ролей, которым доступно изменение фичи из web интерфейса",
    label = "update_feature_web_editable",
    description = """Укажи, каким ролям доступно управление фичей из web-интерфейса.
Доступные роли:
SUPER
SUPERREADER
SUPPORT
PLACER
MANAGER
AGENCY
INTERNAL""",
    consumes = InternalToolsFeatureUpdateWebEditableParam::class,
    type = InternalToolType.WRITER
)
@Action(InternalToolAction.SET)
@Category(InternalToolCategory.FEATURES)
@AccessGroup(InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER)
@ParametersAreNonnullByDefault
class FeatureUpdateWebEditableTool(
    val featureManagingService: FeatureManagingService,
    val featureService: FeatureService,
    val featureCache: FeatureCache,
) : BaseInternalTool<InternalToolsFeatureUpdateWebEditableParam> {

    override fun validate(t: InternalToolsFeatureUpdateWebEditableParam): ValidationResult<InternalToolsFeatureUpdateWebEditableParam, Defect<Any>> {
        val ivb = ItemValidationBuilder.of<InternalToolsFeatureUpdateWebEditableParam, Defect<Any>>(t)
        ivb.item(t.textId, InternalToolsFeatureUpdateWebEditableParam::textId.name)
            .check(notNull())
            .checkBy { featureExists(it) }
        return ivb.result
    }

    @Suppress("UNCHECKED_CAST")
    private fun featureExists(featureTextId: String): ValidationResult<String, Defect<Any>> {
        val feature = featureCache.getCachedByTextId(listOf(featureTextId)).firstOrNull()

        val featureExistsPredicate = Constraint.fromPredicate<String, Defect<Any>>(
            { feature != null },
            CommonDefects.objectNotFound() as Defect<Any>
        )

        return ItemValidationBuilder.of<String, Defect<Any>>(featureTextId)
            .check(featureExistsPredicate)
            .result
    }

    override fun process(p: InternalToolsFeatureUpdateWebEditableParam): InternalToolResult {
        val featureTextId = p.textId
        if (p.rolesAbleToEnable == null) {
            return currentFeatureState(featureTextId)
        }

        val rolesAbleToDisable = p.rolesAbleToDisable ?: p.rolesAbleToEnable
        val result = featureManagingService.setRolesAbleToChangeFeatureFromWeb(
            featureTextId,
            toRoleSet(p.rolesAbleToEnable),
            toRoleSet(rolesAbleToDisable)
        )
        if (ValidationUtils.hasValidationIssues(result)) {
            throw InternalToolValidationException("").withValidationResult(result.getValidationResult())
        }
        return currentFeatureState(featureTextId)
    }

    private fun currentFeatureState(featureTextId: String?): InternalToolResult {
        val feature = featureCache.getCachedByTextId(setOf(featureTextId)).first()
        val message = """Current state for feature $featureTextId:
                    |can enable: ${feature.settings.canEnable}
                    |can disable: ${feature.settings.canDisable}
                """.trimMargin()
        return InternalToolResult()
            .withMessage(message)
    }

    private fun toRoleSet(roles: String?): Set<String> {
        return StreamEx.split(roles, NON_WORD_CHAR)
            .map { it.uppercase() }
            .toSet()
    }

    companion object {
        private val NON_WORD_CHAR = Pattern.compile("[\\W]+")
    }
}
