package ru.yandex.direct.internaltools.tools.feature.tool;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.feature.model.Feature;
import ru.yandex.direct.core.entity.feature.model.FeatureSettings;
import ru.yandex.direct.core.entity.feature.service.FeatureManagingService;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.feature.container.InternalToolsAddFeatureInfo;
import ru.yandex.direct.internaltools.tools.feature.container.InternalToolsAddFeaturesParams;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.staff.client.StaffClient;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeatureConverter.toInternalToolsAddFeatureInfo;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.StringUtils.splitStringToList;
import static ru.yandex.direct.utils.StringUtils.splitStringToSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

@Tool(
        name = "Добавление фичи в базе",
        label = "add_features",
        description = "Добавление одной фичи по ключу.",
        consumes = InternalToolsAddFeaturesParams.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.ADD)
@Category(InternalToolCategory.FEATURES)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class FeaturesAddTool extends MassInternalTool<InternalToolsAddFeaturesParams, InternalToolsAddFeatureInfo> {

    private static final String WATCHERS_SEPARATOR = ",";

    @Autowired
    private FeatureManagingService featureManagingService;

    @Autowired
    private TranslationService translationService;

    @Autowired
    private StaffClient staffClient;

    @Override
    public ValidationResult<InternalToolsAddFeaturesParams, Defect> validate(
            InternalToolsAddFeaturesParams params) {
        ItemValidationBuilder<InternalToolsAddFeaturesParams, Defect> vb =
                ItemValidationBuilder.of(params, Defect.class);

        vb.item(params.getFeatureTextId(), "feature_text_id")
                .check(notNull())
                .check(notBlank(), When.isValid());

        String owner = params.getOwner();
        List<String> watchers = splitStringToList(params.getWatchers(), WATCHERS_SEPARATOR);
        Set<String> existingUsers = getExistingUsers(owner, watchers);

        vb.list(watchers, "watchers")
                .checkEach(inSet(existingUsers));

        vb.item(owner, "owner")
                .check(notBlank())
                .check(inSet(existingUsers));

        return vb.getResult();
    }

    private Set<String> getExistingUsers(String owner, Collection<String> watchers) {
        Set<String> allInputUsers = new HashSet<>(watchers);
        allInputUsers.add(owner);
        return staffClient.getExistingUsers(allInputUsers);
    }

    @Override
    protected List<InternalToolsAddFeatureInfo> getMassData(InternalToolsAddFeaturesParams params) {

        Feature feature = addFeature(params.getFeatureTextId(),
                params.getIsAgencyFeature(),
                params.getClientCanSwitch(),
                splitStringToSet(params.getWatchers(), WATCHERS_SEPARATOR),
                params.getOwner());

        return singletonList(toInternalToolsAddFeatureInfo(feature));
    }

    @Override
    protected List<InternalToolsAddFeatureInfo> getMassData() {
        return emptyList();
    }

    private Feature addFeature(String featureTextId,
                               @Nullable Boolean isAgencyFeature,
                               @Nullable Boolean clientCanSwitch,
                               Set<String> watchers,
                               String owner) {
        Feature feature = getFeature(featureTextId.toLowerCase());

        List<Feature> featuresToAdd = List.of(feature);

        FeatureSettings featureSettings = FeatureManagingService.getDefaultFeatureSettings()
                .withIsAgencyFeature(isAgencyFeature)
                .withOriginalOwner(owner)
                .withOriginalWatchers(watchers);
        if (Boolean.TRUE.equals(clientCanSwitch)) {
            // Добавим роль CLIENT в списки тех, кто может управлять фичей из web
            featureSettings.getCanEnable().add(RbacRole.CLIENT.name());
            featureSettings.getCanDisable().add(RbacRole.CLIENT.name());
        }

        Result<List<Feature>> result = featureManagingService.addFeaturesWithSettings(featuresToAdd, featureSettings);
        if (!result.isSuccessful()) {
            throw new InternalToolValidationException("Can not add feature")
                    .withValidationResult(result.getValidationResult());
        }

        return result.getResult().get(0);
    }

    private Feature getFeature(String featureTextId) {
        Map<String, Feature> existingFeaturesByTextId =
                listToMap(featureManagingService.getCachedFeatures(), Feature::getFeatureTextId);

        if (existingFeaturesByTextId.containsKey(featureTextId)) {
            throw new InternalToolValidationException("This feature already exists in db")
                    .withValidationResult(ValidationResult.failed(featureTextId, invalidValue()));
        }


        FeatureName[] featureNames = FeatureName.values();
        Map<String, Feature> featuresToAddByTextId = StreamEx.of(featureNames)
                .filter(f -> !existingFeaturesByTextId.containsKey(f.getName()))
                .map(f -> new Feature()
                        .withFeatureTextId(f.getName())
                        .withFeaturePublicName(translationService.translate(f.getHumanReadableName())))
                .toMap(Feature::getFeatureTextId, Function.identity());


        if (!featuresToAddByTextId.containsKey(featureTextId)) {
            throw new InternalToolValidationException("Can not find feature with such featureTextId")
                    .withValidationResult(ValidationResult.failed(featureTextId, invalidValue()));
        }

        return featuresToAddByTextId.get(featureTextId);
    }
}
