package ru.yandex.direct.internaltools.tools.feature.tool;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.feature.model.Feature;
import ru.yandex.direct.core.entity.feature.service.FeatureManagingService;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeaturesViewOrDeleteParams;
import ru.yandex.direct.internaltools.tools.feature.container.InternalToolsViewOrDeleteFeatureInfo;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Comparator.comparing;
import static java.util.Comparator.naturalOrder;
import static java.util.Comparator.nullsFirst;
import static java.util.Comparator.nullsLast;
import static java.util.Comparator.reverseOrder;
import static ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeatureConverter.toInternalToolsViewOrDeleteFeatureInfo;
import static ru.yandex.direct.internaltools.tools.feature.container.InternalToolsViewOrDeleteFeatureInfo.TRUE_VALUE;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

@Tool(
        name = "Просмотр и удаление фич в базе",
        label = "view_or_delete_features",
        description = "Список текущих фич. \n"
                + "Удаление одной фичи по ключу. Необходимо, чтобы фича отсутствовала в коде.",
        consumes = InternalToolsFeaturesViewOrDeleteParams.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.EXECUTE)
@Category(InternalToolCategory.FEATURES)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class FeaturesViewOrDeleteTool extends MassInternalTool<InternalToolsFeaturesViewOrDeleteParams,
        InternalToolsViewOrDeleteFeatureInfo> {
    @Autowired
    private FeatureManagingService featureManagingService;

    @Autowired
    private TranslationService translationService;

    @Override
    public ValidationResult<InternalToolsFeaturesViewOrDeleteParams, Defect> validate(
            InternalToolsFeaturesViewOrDeleteParams params) {
        ItemValidationBuilder<InternalToolsFeaturesViewOrDeleteParams, Defect> vb =
                ItemValidationBuilder.of(params, Defect.class);

        InternalToolsFeaturesViewOrDeleteParams.Action action = params.getAction();
        if (action == InternalToolsFeaturesViewOrDeleteParams.Action.DELETE_ONE_FEATURE) {
            vb.item(params.getFeatureTextId(), "feature_text_id")
                    .check(notNull())
                    .check(notBlank(), When.isValid());
        }
        return vb.getResult();
    }

    @Override
    protected List<InternalToolsViewOrDeleteFeatureInfo> getMassData(InternalToolsFeaturesViewOrDeleteParams params) {
        InternalToolsFeaturesViewOrDeleteParams.Action action = params.getAction();
        List<InternalToolsViewOrDeleteFeatureInfo> featureList = getAllFeatureList();

        if (action == InternalToolsFeaturesViewOrDeleteParams.Action.DELETE_ONE_FEATURE) {
            deleteFeature(params.getFeatureTextId());
            featureList = getAllFeatureList();
        }
        return sortResults(featureList, params.getSorting());
    }

    @Override
    protected List<InternalToolsViewOrDeleteFeatureInfo> getMassData() {
        return sortResults(getAllFeatureList(), null);
    }

    private void deleteFeature(String featureTextId) {
        Map<String, Feature> existingFeaturesByTextId =
                listToMap(featureManagingService.getCachedFeatures(), Feature::getFeatureTextId);
        Map<String, FeatureName> featureNameByTextId =
                StreamEx.of(FeatureName.values()).toMap(FeatureName::getName, Function.identity());

        // Удаляем, если фича есть в базе и отсутствует в коде
        if (existingFeaturesByTextId.containsKey(featureTextId) && !featureNameByTextId.containsKey(featureTextId)) {
            Result<Long> result =
                    featureManagingService.deleteFeature(existingFeaturesByTextId.get(featureTextId).getId());
            if (!result.isSuccessful()) {
                throw new InternalToolValidationException("Can not delete feature")
                        .withValidationResult(result.getValidationResult());
            }
        } else {
            throw new InternalToolValidationException("Can not delete feature")
                    .withValidationResult(ValidationResult.failed(featureTextId, invalidValue()));
        }
    }

    private List<InternalToolsViewOrDeleteFeatureInfo> getAllFeatureList() {
        List<InternalToolsViewOrDeleteFeatureInfo> resultList = new ArrayList<>();

        FeatureName[] featureNames = FeatureName.values();
        Map<String, Feature> existingFeaturesByTextId =
                listToMap(featureManagingService.getCachedFeatures(), Feature::getFeatureTextId);

        Map<Long, Map<Boolean, Integer>> summaryForClients = featureManagingService.getFeaturesStateSummary();

        for (FeatureName featureName : featureNames) {
            if (existingFeaturesByTextId.containsKey(featureName.getName())) {
                // Фича есть и в коде, и в базе
                Feature existingFeature = existingFeaturesByTextId.get(featureName.getName());
                resultList.add(toInternalToolsViewOrDeleteFeatureInfo(existingFeature)
                        .withExistsInRepository(TRUE_VALUE)
                        .withExistsInFeatureName(TRUE_VALUE)
                );
                existingFeaturesByTextId.remove(featureName.getName());
            } else {
                // Фича есть только в коде
                resultList.add(toInternalToolsViewOrDeleteFeatureInfo(translationService, featureName)
                        .withExistsInFeatureName(TRUE_VALUE)
                );
            }
        }
        for (Feature feature : existingFeaturesByTextId.values()) {
            // Фича есть только в базе
            InternalToolsViewOrDeleteFeatureInfo info = toInternalToolsViewOrDeleteFeatureInfo(feature)
                    .withExistsInRepository(TRUE_VALUE);
            if (summaryForClients.containsKey(feature.getId())) {
                Map<Boolean, Integer> summary = summaryForClients.get(feature.getId());
                String val = summary.getOrDefault(true, 0).toString() + "/" + summary.getOrDefault(false, 0).toString();
                info.withClientSummary(val);
            }
            resultList.add(info);
        }
        return resultList;
    }

    private List<InternalToolsViewOrDeleteFeatureInfo> sortResults(
            List<InternalToolsViewOrDeleteFeatureInfo> features,
            @Nullable InternalToolsFeaturesViewOrDeleteParams.SortType sortType) {
        Comparator<InternalToolsViewOrDeleteFeatureInfo> comparator = getComparator(sortType);
        return StreamEx.of(features)
                .sorted(comparator)
                .toList();
    }

    private Comparator<InternalToolsViewOrDeleteFeatureInfo> getComparator(
            @Nullable InternalToolsFeaturesViewOrDeleteParams.SortType sortType) {
        Comparator<InternalToolsViewOrDeleteFeatureInfo> defaultComparator =
                comparing(InternalToolsViewOrDeleteFeatureInfo::getFeatureId, nullsFirst(naturalOrder()));
        if (sortType == null) {
            return defaultComparator;
        }
        Comparator<InternalToolsViewOrDeleteFeatureInfo> comparator;
        switch (sortType) {
            case BY_TEXT_ID:
                comparator = comparing(InternalToolsViewOrDeleteFeatureInfo::getFeatureTextId);
                break;
            case BY_EXISTING_IN_DB:
                comparator =
                        comparing(InternalToolsViewOrDeleteFeatureInfo::getExistsInRepository,
                                nullsFirst(reverseOrder()));
                break;
            case BY_EXISTING_IN_CODE:
                comparator =
                        comparing(InternalToolsViewOrDeleteFeatureInfo::getExistsInFeatureName,
                                nullsFirst(reverseOrder()));
                break;
            case BY_PERCENTS:
                comparator = comparing(InternalToolsViewOrDeleteFeatureInfo::getIsTurnOn, nullsLast(reverseOrder()));
                break;
            default:
                return defaultComparator;
        }
        return comparator.thenComparing(defaultComparator);
    }
}
