package ru.yandex.direct.internaltools.tools.freelancer.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.freelancer.model.Freelancer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCard;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificate;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificateType;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.internaltools.tools.freelancer.model.IntToolFreelancerCard;
import ru.yandex.direct.rbac.RbacService;

import static ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificateType.DIRECT;
import static ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificateType.METRIKA;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class IntToolFreelancerConverterService {

    private final UserService userService;
    private final RbacService rbacService;

    public IntToolFreelancerConverterService(UserService userService, RbacService rbacService) {
        this.userService = userService;
        this.rbacService = rbacService;
    }

    /**
     * Преобразует список {@link Freelancer} в {@link IntToolFreelancerCard}, обогощая данные ФИО фрилансера и его валютой
     */
    public List<IntToolFreelancerCard> getIntToolFreelancerCards(List<Freelancer> freelancers) {
        Map<ClientId, Long> chiefsByClientIds =
                rbacService.getChiefsByClientIds(mapList(freelancers, f -> ClientId.fromLong(f.getId())));
        Collection<User> users = userService.massGetUser(chiefsByClientIds.values());
        Map<Long, User> usersByUid = listToMap(users, User::getUid);
        return StreamEx.of(freelancers)
                .mapToEntry(Freelancer::getId)
                .mapValues(ClientId::fromLong)
                .mapValues(chiefsByClientIds::get)
                .mapValues(usersByUid::get)
                .mapKeyValue(this::convert)
                .toList();
    }

    private IntToolFreelancerCard convert(Freelancer freelancer, User chief) {
        IntToolFreelancerCard card = new IntToolFreelancerCard();
        card.setClientId(freelancer.getFreelancerId());
        FreelancerCard freelancerCard = freelancer.getCard();
        if (freelancerCard != null) {
            card.setClientAvatarId(freelancerCard.getAvatarId());
        }
        List<FreelancerCertificate> certificates = freelancer.getCertificates();
        card.setSearchable(freelancer.getIsSearchable());
        card.setFirstName(freelancer.getFirstName());
        card.setSecondName(freelancer.getSecondName());
        card.setLogin(chief.getLogin());
        card.setCertLogin(freelancer.getCertLogin());
        if (certificates != null && !certificates.isEmpty()) {
            Map<FreelancerCertificateType, Long> certIdByType = StreamEx.of(certificates)
                    .mapToEntry(FreelancerCertificate::getType, FreelancerCertificate::getCertId)
                    .distinctKeys()
                    .toMap();
            card.setDirectCertId(certIdByType.get(DIRECT));
            card.setMetrikaCertId(certIdByType.get(METRIKA));
        }
        return card;
    }
}
