package ru.yandex.direct.internaltools.tools.freelancer.tool;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter;
import ru.yandex.direct.core.entity.freelancer.model.ClientAvatarId;
import ru.yandex.direct.core.entity.freelancer.model.Freelancer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerBase;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCard;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCardModeration;
import ru.yandex.direct.core.entity.freelancer.model.FreelancersCardStatusModerate;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerCardService;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerClientAvatarService;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.freelancer.container.IntToolCardModerationConverter;
import ru.yandex.direct.internaltools.tools.freelancer.model.CardModerationParameters;
import ru.yandex.direct.internaltools.tools.freelancer.model.IntToolCardModeration;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter.enabledFreelancers;
import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.clientNotFound;
import static ru.yandex.direct.core.entity.freelancer.utils.ClientAvatarIdUtils.clientAvatarIdFromCards;
import static ru.yandex.direct.internaltools.tools.freelancer.container.IntToolFeedbackModerationConverter.convertToCardModerationResult;
import static ru.yandex.direct.internaltools.tools.freelancer.model.CardModerationParameters.CARD_ID;
import static ru.yandex.direct.internaltools.tools.freelancer.model.CardModerationParameters.DECLINE_REASON;
import static ru.yandex.direct.internaltools.tools.freelancer.model.CardModerationParameters.FREELANCER_ID;
import static ru.yandex.direct.internaltools.tools.freelancer.model.CardModerationParameters.FREELANCER_LOGIN;
import static ru.yandex.direct.internaltools.tools.freelancer.model.CardModerationParameters.MODERATION_RESULT;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Tool(
        name = "3 Модерировать карточки фрилансеров",
        label = "card_moderation",
        description =
                "В списке ниже только карточки ожидающие модерации.  \nЕсли указать freelancerId или freelancerLogin, то отчёт покажет только карточку указанного фрилансера. \n"
                        + "Для изменения статуса карточки необходимо указать оба значения: freelancerId и cardId.",
        consumes = CardModerationParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.FREELANCER)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.SUPERREADER, InternalToolAccessRole.MANAGER,
        InternalToolAccessRole.SUPPORT, InternalToolAccessRole.PLACER})
@ParametersAreNonnullByDefault
public class FreelancerCardModerationTool extends MassInternalTool<CardModerationParameters, IntToolCardModeration> {

    private final FreelancerCardService freelancerCardService;
    private final FreelancerService freelancerService;
    private final UserService userService;
    private final FreelancerClientAvatarService freelancerClientAvatarService;

    @Autowired
    public FreelancerCardModerationTool(
            FreelancerCardService freelancerCardService,
            FreelancerService freelancerService, UserService userService,
            FreelancerClientAvatarService freelancerClientAvatarService) {
        this.freelancerCardService = freelancerCardService;
        this.freelancerService = freelancerService;
        this.userService = userService;
        this.freelancerClientAvatarService = freelancerClientAvatarService;
    }

    @Override
    public ValidationResult<CardModerationParameters, Defect> validate(
            CardModerationParameters params) {
        ItemValidationBuilder<CardModerationParameters, Defect> vb =
                ItemValidationBuilder.of(params, Defect.class);
        vb.item(params.getFreelancerId(), FREELANCER_ID)
                .check(notNull(), When.isTrue(params.getFreelancerLogin() == null))
                .check(validId(), When.isValid());
        vb.item(params.getFreelancerLogin(), FREELANCER_LOGIN)
                .check(notNull(), When.isTrue(params.getFreelancerId() == null))
                .check(notBlank(), When.isValid());
        if (params.getCardId() != null || params.getModerationResult() != null) {
            vb.item(params.getCardId(), CARD_ID)
                    .check(notNull())
                    .check(validId());
            vb.item(params.getModerationResult(), MODERATION_RESULT)
                    .check(notNull());
            if (params.getModerationResult() != null &&
                    params.getModerationResult().equals(FreelancersCardStatusModerate.DECLINED.name())) {
                vb.item(params.getDeclineReason(), DECLINE_REASON)
                        .check(notNull())
                        .check(notEmptyCollection(), When.isValid());
            }
        }
        return vb.getResult();
    }

    @Nullable
    @Override
    protected List<IntToolCardModeration> getMassData() {
        List<Long> allSearchableFreelancerIds = getAllSearchableFreelancerIds();
        return getIntToolFeedbackModerationList(allSearchableFreelancerIds);
    }

    private List<Long> getAllSearchableFreelancerIds() {
        FreelancersQueryFilter filter = enabledFreelancers().build();
        List<Freelancer> allSearchableFreelancers = freelancerService.getFreelancers(filter);
        return StreamEx.of(allSearchableFreelancers)
                .map(FreelancerBase::getId)
                .toList();
    }

    private List<IntToolCardModeration> getIntToolFeedbackModerationList(List<Long> freelancerIds) {
        Map<Long, String> loginByClientId = getLogins(freelancerIds);
        Map<Long, String> namesByClientId = getNames(loginByClientId);
        List<FreelancerCard> newestFreelancerCards = freelancerCardService.getNewestFreelancerCards(freelancerIds);
        List<FreelancerCard> notModeratedCards = StreamEx.of(newestFreelancerCards)
                .filter(card -> !card.getIsArchived()
                        && !card.getStatusModerate().equals(FreelancersCardStatusModerate.ACCEPTED)
                        && !card.getStatusModerate().equals(FreelancersCardStatusModerate.DECLINED))
                .toList();
        Map<Long, String> avatarUrlByClientId = getAvatarUrls(notModeratedCards);
        return StreamEx.of(notModeratedCards)
                .map(card -> IntToolCardModerationConverter
                        .convertCardToIntToolParameter(card, loginByClientId, namesByClientId, avatarUrlByClientId))
                .toList();
    }

    private Map<Long, String> getLogins(List<Long> freelancerIds) {
        List<ClientId> clientIds = StreamEx.of(freelancerIds)
                .map(ClientId::fromLong)
                .toList();
        Map<ClientId, String> chiefsLoginsByClientIds = userService.getChiefsLoginsByClientIds(clientIds);
        return EntryStream.of(chiefsLoginsByClientIds)
                .mapKeys(ClientId::asLong)
                .toMap();
    }

    // Этот метод специально возвращает только имена, а не юзеров.
    // Когда появится специальное поле с публичным именем фрилансера, этот метод станет не нужен и его можно будет просто удалить.
    private Map<Long, String> getNames(Map<Long, String> loginByClientId) {
        Collection<String> userLogins = loginByClientId.values();
        Collection<User> users = userService.massGetUserByLogin(userLogins);
        return StreamEx.of(users)
                .toMap(user -> user.getClientId().asLong(), User::getFio);
    }

    private Map<Long, String> getAvatarUrls(List<FreelancerCard> freelancerCards) {
        List<ClientAvatarId> avatarIdByClientId = clientAvatarIdFromCards(freelancerCards);
        return freelancerClientAvatarService.massGetUrlSize180(avatarIdByClientId);
    }

    @Override
    protected List<IntToolCardModeration> getMassData(CardModerationParameters params) {
        Long freelancerId = getFreelancerId(params);
        if (params.getCardId() == null) {
            return getIntToolFeedbackModerationList(singletonList(freelancerId));
        }
        FreelancerCardModeration cardModerationResult =
                convertToCardModerationResult(params, freelancerId);
        Result<Long> result = freelancerCardService.applyModerationResult(cardModerationResult);
        ValidationResult<?, Defect> validationResult = result.getValidationResult();
        if (validationResult.hasAnyErrors()) {
            throw new InternalToolValidationException("Freelancer card was not updated")
                    .withValidationResult(validationResult);
        }
        return getMassData();
    }

    private Long getFreelancerId(CardModerationParameters params) {
        if (params.getFreelancerId() != null) {
            return params.getFreelancerId();
        }
        String freelancerLogin = params.getFreelancerLogin();
        // Согласно валидации, если freelancerId == null, то freelancerLogin не может быть null
        User freelancerUser = userService.getUserByLogin(freelancerLogin);
        if (freelancerUser != null) {
            return freelancerUser.getClientId().asLong();
        }
        // Фрилансер с таким логином не найден
        throw new InternalToolValidationException("Can not find freelancer")
                .withValidationResult(ValidationResult.failed(freelancerLogin, clientNotFound()));
    }
}
