package ru.yandex.direct.internaltools.tools.freelancer.tool;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter;
import ru.yandex.direct.core.entity.freelancer.model.Freelancer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerBase;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificate;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificateType;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerService;
import ru.yandex.direct.core.entity.freelancer.service.utils.FreelancerCertificateWithNames;
import ru.yandex.direct.core.entity.freelancer.service.utils.FreelancerCertificatesUtils;
import ru.yandex.direct.core.entity.user.service.BlackboxUserService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.expert.client.ExpertClient;
import ru.yandex.direct.expert.client.model.Certificate;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolParameter;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.freelancer.model.CertTrouble;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.utils.CollectionUtils;

import static java.util.Collections.emptySet;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificateType.DIRECT;
import static ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificateType.DIRECT_PRO;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Tool(
        name = "4 Показывает истёкшие типы сертификатов у фрилансеров",
        label = "freelancer_expired_certs",
        description = "Программа смотрит, какого типа сертификаты есть у фрилансера в базе Директа, потом идёт в " +
                "Сетрификатницу и проверяет, есть ли на аккаунтах фрилансера активные (не просроченные) сертификаты " +
                "этих же типов. Если на какой-то из типов активного сертификата не находится, то фрилансер попадает " +
                "в список ниже. При наличии сертификата с типом DIRECT_PRO, сертификат с типом DIRECT считается " +
                "не нужным, в обратную сторону - нет.",
        consumes = InternalToolParameter.class,
        type = InternalToolType.REPORT
)
@Action(InternalToolAction.SHOW)
@Category(InternalToolCategory.FREELANCER)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.MANAGER,
        InternalToolAccessRole.SUPERREADER, InternalToolAccessRole.SUPPORT, InternalToolAccessRole.PLACER})
@ParametersAreNonnullByDefault
public class FreelancerExpiredCertsTool extends MassInternalTool<InternalToolParameter, CertTrouble> {

    private final FreelancerService freelancerService;
    private final UserService userService;
    private final ExpertClient expertClient;
    private final BlackboxUserService blackboxUserService;
    private final RbacService rbacService;

    public FreelancerExpiredCertsTool(
            FreelancerService freelancerService,
            UserService userService,
            ExpertClient expertClient,
            BlackboxUserService blackboxUserService,
            RbacService rbacService) {
        this.freelancerService = freelancerService;
        this.userService = userService;
        this.expertClient = expertClient;
        this.blackboxUserService = blackboxUserService;
        this.rbacService = rbacService;
    }

    /**
     * Удаляет дубликаты из листа по заданному ключу.
     */
    private static <T> List<T> distinctList(Iterable<T> source, Function<? super T, ?> keyExtractor) {
        return StreamEx.of(source.spliterator())
                .distinct(keyExtractor)
                .toList();
    }

    @Override
    protected List<CertTrouble> getMassData(@Nullable InternalToolParameter params) {
        List<Freelancer> freelancers = getFreelancers();
        Map<Long, CertTrouble> troubleById =
                listToMap(freelancers, FreelancerBase::getId, f -> new CertTrouble().withId(f.getId()));

        Map<Long, Long> freelancerIdByUid = getFreelancerIdByUid(freelancers, troubleById);
        Map<Long, List<FreelancerCertificateWithNames>> extCertsByFreelancerId = getExtCerts(freelancerIdByUid);
        fillExpiredCertTrouble(freelancers, extCertsByFreelancerId, troubleById);
        return getNotEmptyTrobles(troubleById);
    }

    private List<CertTrouble> getNotEmptyTrobles(Map<Long, CertTrouble> troubleById) {
        Map<Long, CertTrouble> notEmptyTroubleById = EntryStream.of(troubleById)
                .filter(entry -> isNotBlank(entry.getValue().getExpiredCertTypes()))
                .toMap();
        List<ClientId> clientIds = mapList(notEmptyTroubleById.keySet(), ClientId::fromLong);
        Map<ClientId, String> loginByClientId = userService.getChiefsLoginsByClientIds(clientIds);
        loginByClientId.forEach((clientId, login) -> notEmptyTroubleById.get(clientId.asLong()).withLogin(login));
        return StreamEx.of(notEmptyTroubleById.values())
                .sortedByLong(CertTrouble::getId)
                .toList();
    }

    private Map<Long, Long> getFreelancerIdByUid(List<Freelancer> freelancers, Map<Long, CertTrouble> troubleById) {
        List<ClientId> clientIds = mapList(troubleById.keySet(), ClientId::fromLong);
        Map<ClientId, Long> chiefsByClientIds = rbacService.getChiefsByClientIds(clientIds);
        Map<Long, Long> freelancerIdByUid = EntryStream.of(chiefsByClientIds)
                .invert()
                .mapValues(ClientId::asLong)
                .toMap();

        List<Freelancer> frlWithCertLogin = filterList(freelancers, f -> isNotBlank(f.getCertLogin()));
        for (Freelancer frl : frlWithCertLogin) {
            String certLogin = frl.getCertLogin();
            Optional<Long> certUid = blackboxUserService.getUidByLogin(certLogin);
            Long freelancerId = frl.getFreelancerId();
            if (certUid.isPresent()) {
                freelancerIdByUid.put(certUid.get(), freelancerId);
            } else {
                CertTrouble trouble = troubleById.get(freelancerId);
                String errorMessage = "Логин сертификата '" + certLogin + "' не " + "найден";
                appendDescription(trouble, errorMessage);
            }
        }
        return freelancerIdByUid;
    }

    private void appendDescription(CertTrouble trouble, String str) {
        String expiredCertTypes = nvl(trouble.getExpiredCertTypes(), "");
        expiredCertTypes += str + "; ";
        trouble.withExpiredCertTypes(expiredCertTypes);
    }

    private void fillExpiredCertTrouble(List<Freelancer> freelancers,
                                        Map<Long, List<FreelancerCertificateWithNames>> extCertsByFreelancerId,
                                        Map<Long, CertTrouble> troubleById) {

        Map<Long, Set<FreelancerCertificateType>> extTypesByFreelancerId = EntryStream.of(extCertsByFreelancerId)
                .mapValues(certs -> listToSet(certs, l -> l.getFreelancerCertificate().getType()))
                .toMap();

        Map<Long, List<FreelancerCertificateType>> localTypesByFreelancerId = StreamEx.of(freelancers)
                .mapToEntry(FreelancerBase::getFreelancerId, FreelancerBase::getCertificates)
                .mapValues(certs -> StreamEx.of(certs)
                        .map(FreelancerCertificate::getType)
                        .distinct()
                        .sorted()
                        .toList())
                .toMap();

        for (Map.Entry<Long, List<FreelancerCertificateType>> entry : localTypesByFreelancerId.entrySet()) {
            Long freelancerId = entry.getKey();
            List<FreelancerCertificateType> localCertTypes = entry.getValue();
            Set<FreelancerCertificateType> extCertTypes = extTypesByFreelancerId.getOrDefault(freelancerId, emptySet());
            CertTrouble trouble = troubleById.get(freelancerId);
            compareLocalAndExternalCerts(trouble, localCertTypes, extCertTypes);
        }
    }

    private void compareLocalAndExternalCerts(CertTrouble freelancerTrouble,
                                              List<FreelancerCertificateType> localCertTypes,
                                              Set<FreelancerCertificateType> extCertTypes) {
        for (FreelancerCertificateType localCertType : localCertTypes) {
            // Приравниваем наличие сертификата DIRECT_PRO к наличию сертификата DIRECT, т.к. с июля 2019-го
            // первый был заменён вторым
            if (extCertTypes.contains(localCertType)
                    || (localCertType == DIRECT && extCertTypes.contains(DIRECT_PRO))) {
                continue;
            }
            appendDescription(freelancerTrouble, localCertType.name());
        }
    }

    /**
     * Ходит в сертификатницу и получает для фрилансеров все их актуальные сертификаты.
     *
     * @param freelancerIdByUid Мапа, в ключе - uid фрилансера или аккаунта к которому привязаны сертификаты, в
     *                          значении - ClientId фрилансера.
     * @return Мапа, в ключе - ClientId фрилансера, в значении - список всех активныех сертификатов фрилансера
     * показываемыех в интерфейсе Директа.
     */
    private Map<Long, List<FreelancerCertificateWithNames>> getExtCerts(Map<Long, Long> freelancerIdByUid) {
        Map<Long, List<Certificate>> extCertsByUid =
                expertClient.getCertificates(List.copyOf(freelancerIdByUid.keySet()));

        Map<Long, List<List<FreelancerCertificateWithNames>>> extCertsGrouping = EntryStream.of(extCertsByUid)
                .mapKeys(freelancerIdByUid::get)
                .mapValues(FreelancerCertificatesUtils::convertSupportedCertificates)
                .grouping();

        return EntryStream.of(extCertsGrouping)
                .mapValues(CollectionUtils::flatToList)
                .mapValues(certs -> distinctList(certs, cert -> cert.getFreelancerCertificate().getCertId()))
                .toMap();
    }

    private List<Freelancer> getFreelancers() {
        FreelancersQueryFilter queryFilter = FreelancersQueryFilter.enabledFreelancers().build();
        return freelancerService.getFreelancers(queryFilter);
    }

}
