package ru.yandex.direct.internaltools.tools.freelancer.tool;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.freelancer.model.Freelancer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerBase;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerFeedback;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerUgcDeclineReason;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerUgcModerationStatus;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerFeedbackService;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.freelancer.container.IntToolFeedbackModerationConverter;
import ru.yandex.direct.internaltools.tools.freelancer.model.FeedbackModerationParameters;
import ru.yandex.direct.internaltools.tools.freelancer.model.IntToolFeedbackModeration;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter.enabledFreelancers;
import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.clientNotFound;
import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.feedbackNotFound;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.internaltools.tools.freelancer.model.FeedbackModerationParameters.DECLINE_REASON;
import static ru.yandex.direct.internaltools.tools.freelancer.model.FeedbackModerationParameters.FEEDBACK_ID;
import static ru.yandex.direct.internaltools.tools.freelancer.model.FeedbackModerationParameters.FREELANCER_ID;
import static ru.yandex.direct.internaltools.tools.freelancer.model.FeedbackModerationParameters.FREELANCER_LOGIN;
import static ru.yandex.direct.internaltools.tools.freelancer.model.FeedbackModerationParameters.MODERATION_RESULT;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Tool(
        name = "Модерация отзывов на фрилансера",
        label = "feedback_moderation",
        description =
                "Если указать только freelancerId или freelancerLogin, то отчёт покажет все отзывы на указанного фрилансера. \n"
                        + "Для модерации достаточно указывать одно из значений freelancerId или freelancerLogin.",
        consumes = FeedbackModerationParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.FREELANCER)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.SUPERREADER, InternalToolAccessRole.MANAGER,
        InternalToolAccessRole.SUPPORT})
@ParametersAreNonnullByDefault
public class FreelancerFeedbackModerationTool extends MassInternalTool<FeedbackModerationParameters, IntToolFeedbackModeration> {

    private final FreelancerFeedbackService feedbackService;
    private final FreelancerService freelancerService;
    private final UserService userService;

    @Autowired
    public FreelancerFeedbackModerationTool(
            FreelancerFeedbackService feedbackService,
            FreelancerService freelancerService, UserService userService) {
        this.feedbackService = feedbackService;
        this.freelancerService = freelancerService;
        this.userService = userService;
    }

    @Override
    public ValidationResult<FeedbackModerationParameters, Defect> validate(
            FeedbackModerationParameters params) {
        ItemValidationBuilder<FeedbackModerationParameters, Defect> vb =
                ItemValidationBuilder.of(params, Defect.class);
        vb.item(params.getFreelancerId(), FREELANCER_ID)
                .check(notNull(), When.isTrue(params.getFreelancerLogin() == null))
                .check(validId(), When.isValid());
        vb.item(params.getFreelancerLogin(), FREELANCER_LOGIN)
                .check(notNull(), When.isTrue(params.getFreelancerId() == null))
                .check(notBlank(), When.isValid());
        if (params.getFeedbackId() != null) {
            vb.item(params.getFeedbackId(), FEEDBACK_ID)
                    .check(notBlank());
            vb.item(params.getModerationResult(), MODERATION_RESULT)
                    .check(notNull());
            if (params.getModerationResult() != null &&
                    params.getModerationResult().equals(FreelancerUgcModerationStatus.DECLINED.name())) {
                vb.item(params.getDeclineReason(), DECLINE_REASON)
                        .check(notNull());
            }
        }
        return vb.getResult();
    }

    @Override
    protected List<IntToolFeedbackModeration> getMassData(FeedbackModerationParameters params) {
        Long freelancerId = params.getFreelancerId();
        String feedbackId = params.getFeedbackId();
        String freelancerLogin = params.getFreelancerLogin();

        if (freelancerId == null) {
            // Согласно валидации, если freelancerId == null, то freelancerLogin не может быть null
            User freelancerUser = userService.getUserByLogin(freelancerLogin);
            if (freelancerUser != null) {
                freelancerId = freelancerUser.getClientId().asLong();
            } else {
                // Фрилансер с таким логином не найден
                throw new InternalToolValidationException("Can not find freelancer")
                        .withValidationResult(ValidationResult.failed(freelancerLogin, clientNotFound()));
            }
        }

        if (feedbackId == null) {
            return getIntToolFeedbackModerationByFreelancerId(freelancerId);
        }

        FreelancerFeedback feedback = feedbackService.getFreelancerFeedback(feedbackId, freelancerId);
        if (feedback == null) {
            throw new InternalToolValidationException("Can not find feedback")
                    .withValidationResult(ValidationResult.failed(feedbackId, feedbackNotFound()));
        }
        FreelancerUgcModerationStatus moderationStatus =
                FreelancerUgcModerationStatus.valueOf(params.getModerationResult());
        FreelancerUgcDeclineReason declineReason = null;
        FreelancerFeedback updatedFeedback = feedback.withModerationStatus(moderationStatus);
        if (moderationStatus.equals(FreelancerUgcModerationStatus.DECLINED)) {
            declineReason = FreelancerUgcDeclineReason.valueOf(params.getDeclineReason());
            updatedFeedback.withDeclineReasonCode(declineReason);
        }
        Result<String> result = feedbackService
                .updateFreelancerFeedbackModeration(feedbackId, freelancerId, moderationStatus, declineReason);
        if (hasValidationIssues(result)) {
            throw new InternalToolValidationException("Can not update freelancer")
                    .withValidationResult(result.getValidationResult());
        }
        return getMassData();
    }

    @Nullable
    @Override
    protected List<IntToolFeedbackModeration> getMassData() {
        List<Freelancer> freelancers = freelancerService.getFreelancers(enabledFreelancers().build());
        return getIntToolFeedbackModerationList(freelancers);
    }

    private List<IntToolFeedbackModeration> getIntToolFeedbackModerationByFreelancerId(Long freelancerId) {
        List<Freelancer> freelancers = freelancerService.getFreelancers(singletonList(freelancerId));
        return getIntToolFeedbackModerationList(freelancers);
    }

    private List<IntToolFeedbackModeration> getIntToolFeedbackModerationList(List<Freelancer> freelancers) {
        List<IntToolFeedbackModeration> moderationList = freelancers.stream()
                .map(FreelancerBase::getFreelancerId)
                .map(feedbackService::getFreelancerFeedbackList)
                .flatMap(Collection::stream)
                .map(IntToolFeedbackModerationConverter::convertToIntToolFeedbackModeration)
                .collect(Collectors.toList());

        List<Long> authorUids = StreamEx.of(moderationList)
                .map(IntToolFeedbackModeration::getAuthorUid)
                .toList();
        Map<Long, String> authorLoginsByUid = StreamEx.of(userService.massGetUser(authorUids))
                .toMap(User::getUid, User::getLogin);

        List<ClientId> clientIds = StreamEx.of(moderationList)
                .map(IntToolFeedbackModeration::getFreelancerId)
                .map(ClientId::fromLong)
                .toList();
        Map<ClientId, String> chiefsLoginsByClientIds = userService.getChiefsLoginsByClientIds(clientIds);

        moderationList.forEach(item ->
                item.withAuthorLogin(authorLoginsByUid.get(item.getAuthorUid()))
                        .withFreelancerLogin(chiefsLoginsByClientIds.get(ClientId.fromLong(item.getFreelancerId()))));
        return moderationList;
    }
}
