package ru.yandex.direct.internaltools.tools.freelancer.tool;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.freelancer.model.Freelancer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificate;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificateType;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerRegisterService;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerService;
import ru.yandex.direct.core.entity.freelancer.service.utils.FreelancerCertificateWithNames;
import ru.yandex.direct.core.entity.freelancer.service.utils.FreelancerCertificatesUtils;
import ru.yandex.direct.core.entity.user.service.BlackboxUserService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.expert.client.ExpertClient;
import ru.yandex.direct.expert.client.model.Certificate;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.freelancer.model.FreelancerRegisterParameters;
import ru.yandex.direct.internaltools.tools.freelancer.model.IntToolFreelancerCard;
import ru.yandex.direct.internaltools.tools.freelancer.service.IntToolFreelancerConverterService;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.sender.YandexSenderClient;
import ru.yandex.direct.sender.YandexSenderTemplateParams;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter.enabledFreelancers;
import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.clientIsAlreadyFreelancer;
import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.clientNotFound;
import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.directCertificateNotFound;
import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.mustBeClient;
import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.namesMustBeSameAsCertNames;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.dbutil.model.ClientId.fromLong;
import static ru.yandex.direct.validation.Predicates.isDouble;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.unconditional;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Tool(
        name = "Регистрация фрилансера",
        label = "register_freelancer",
        description = "Регистрация существующего пользователя как фрилансера",
        consumes = FreelancerRegisterParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.ADD)
@Category(InternalToolCategory.FREELANCER)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.MANAGER,
        InternalToolAccessRole.SUPPORT})
@ParametersAreNonnullByDefault
public class FreelancerRegisterTool extends MassInternalTool<FreelancerRegisterParameters, IntToolFreelancerCard> {

    private static final int MAX_NAME_LENGTH = 50;
    private static final String CLIENT_ID = "ClientID";

    private final FreelancerRegisterService freelancerRegisterService;
    private final FreelancerService freelancerService;
    private final IntToolFreelancerConverterService converterService;
    private final ClientService clientService;
    private final UserService userService;
    private final ExpertClient expertClient;
    private final YandexSenderClient senderClient;
    private final String senderRegisterFreelancerSlug;
    private final BlackboxUserService blackboxUserService;

    public FreelancerRegisterTool(
            FreelancerRegisterService freelancerRegisterService,
            FreelancerService freelancerService,
            IntToolFreelancerConverterService converterService,
            ClientService clientService,
            UserService userService,
            ExpertClient expertClient,
            YandexSenderClient senderClient,
            @Value("${freelancers.sender_keys.register_freelancer}") String senderRegisterFreelancerSlug,
            BlackboxUserService blackboxUserService) {
        this.freelancerRegisterService = freelancerRegisterService;
        this.freelancerService = freelancerService;
        this.converterService = converterService;
        this.clientService = clientService;
        this.userService = userService;
        this.expertClient = expertClient;
        this.senderClient = senderClient;
        this.senderRegisterFreelancerSlug = senderRegisterFreelancerSlug;
        this.blackboxUserService = blackboxUserService;
    }

    @Override
    public ValidationResult<FreelancerRegisterParameters, Defect> validate(FreelancerRegisterParameters params) {
        ItemValidationBuilder<FreelancerRegisterParameters, Defect> vb = ItemValidationBuilder.of(params, Defect.class);
        vb.item(params.getFirstName(), "firstName")
                .check(notNull())
                .check(notBlank())
                .check(maxStringLength(MAX_NAME_LENGTH));
        vb.item(params.getSecondName(), "secondName")
                .check(notNull())
                .check(notBlank())
                .check(maxStringLength(MAX_NAME_LENGTH));

        ItemValidationBuilder<String, Defect> ratingVb = vb.item(params.getInitialRating(), "initialRating");
        ratingVb.check(Constraint.fromPredicate(isDouble(), CommonDefects.invalidValue()));
        if (!ratingVb.getResult().hasAnyErrors()) {
            ratingVb.item(new BigDecimal(params.getInitialRating()), "num")
                    .check(inRange(BigDecimal.ZERO, BigDecimal.valueOf(5)));
        }
        if (ratingVb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }

        Long uid = userService.getUidByLogin(params.getLogin());
        ItemValidationBuilder<String, Defect> loginVb = vb.item(params.getLogin(), "login");
        loginVb.check(unconditional(clientNotFound()), When.isTrue(uid == null));
        if (loginVb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }

        Map<Long, Client> clients = clientService.massGetClientsByUids(singletonList(uid));
        Client client = clients.get(uid);
        loginVb.checkBy(validateLogin(client));
        params.setClient(client);

        String certLogin = params.getCertLogin();
        if (isNotBlank(certLogin)) {
            ItemValidationBuilder<String, Defect> certLoginVb = vb.item(certLogin, "certLogin");
            Long certUid = blackboxUserService.getUidByLogin(certLogin).orElse(null);
            certLoginVb.check(unconditional(objectNotFound()), When.isTrue(certUid == null));
            params.setCertUid(certUid);
        }
        return vb.getResult();
    }

    @Override
    protected List<IntToolFreelancerCard> getMassData(FreelancerRegisterParameters params) {
        List<Long> uids = new ArrayList<>();
        Long selfCertUid = params.getClient().getChiefUid();
        uids.add(selfCertUid);
        Long extCertUid = params.getCertUid();
        if (extCertUid != null) {
            uids.add(extCertUid);
        }
        Map<Long, List<Certificate>> returnedCerts = expertClient.getCertificates(uids);
        Map<Long, List<FreelancerCertificateWithNames>> supportedCertificates =
                EntryStream.of(returnedCerts)
                        .mapValues(FreelancerCertificatesUtils::convertSupportedCertificates)
                        .toMap();

        boolean isNoneDirectTypeCertificate = StreamEx.ofValues(supportedCertificates)
                .flatMap(Collection::stream)
                .noneMatch(FreelancerCertificatesUtils::isDirectType);
        if (isNoneDirectTypeCertificate) {
            throw new InternalToolValidationException("")
                    .withValidationResult(ValidationResult.failed(params.getLoginForCert(),
                            directCertificateNotFound()));
        }

        List<FreelancerCertificateWithNames> extOnlySameNames;
        if (extCertUid != null) {
            List<FreelancerCertificateWithNames> extCerts = supportedCertificates.getOrDefault(extCertUid, emptyList());
            extOnlySameNames = StreamEx.of(extCerts)
                    .filter(c -> FreelancerCertificatesUtils.areNamesEqual(c, params.getFirstName(),
                            params.getSecondName()))
                    .toList();
            if (!extCerts.isEmpty() && extOnlySameNames.isEmpty()) {
                throw new InternalToolValidationException("")
                        .withValidationResult(ValidationResult.failed(params.getLoginForCert(),
                                namesMustBeSameAsCertNames()));
            }
        } else {
            extOnlySameNames = emptyList();
        }

        List<FreelancerCertificateWithNames> selfCertsWithNames = supportedCertificates.get(selfCertUid);
        Map<FreelancerCertificateType, List<FreelancerCertificateWithNames>> certsByType =
                StreamEx.of(selfCertsWithNames)
                        .append(extOnlySameNames)
                        .groupingBy(cert -> cert.getFreelancerCertificate().getType());
        List<FreelancerCertificate> allCertificates = StreamEx.of(certsByType.values())
                .filter(CollectionUtils::isNotEmpty)
                .map(StreamEx::of)
                //Оставляем только новейшие сертификаты одного типа
                .map(l -> l.maxBy(FreelancerCertificateWithNames::getConfirmedDate).orElseThrow())
                .map(FreelancerCertificateWithNames::getFreelancerCertificate)
                .toList();

        Freelancer freelancer = new Freelancer();
        long clientId = params.getClient().getClientId();
        freelancer.withFirstName(params.getFirstName())
                .withSecondName(params.getSecondName())
                .withCertLogin(params.getCertLogin())
                .withFreelancerId(clientId)
                .withId(clientId)
                .withCertificates(allCertificates)
                .withIsSearchable(true)
                .withRating(Double.valueOf(params.getInitialRating()));

        Long operatorUid = params.getOperator().getUid();
        Result<Freelancer> operationResult =
                freelancerRegisterService.registerFreelancer(fromLong(clientId), operatorUid, freelancer);

        if (hasValidationIssues(operationResult)) {
            throw new InternalToolValidationException("").withValidationResult(operationResult.getValidationResult());
        }

        YandexSenderTemplateParams templateParams = buildTemplateParams(params);
        senderClient.sendTemplate(templateParams);

        return getMassData();
    }

    private YandexSenderTemplateParams buildTemplateParams(FreelancerRegisterParameters params) {
        return new YandexSenderTemplateParams.Builder()
                .withCampaignSlug(senderRegisterFreelancerSlug)
                .withToEmail(params.getEmail())
                .withAsync(Boolean.TRUE)
                .withArgs(ImmutableMap.of(
                        "first_name", params.getFirstName(),
                        "edit_link",
                        "https://" + params.getCurrentHost() + "/dna/freelancers/" + params.getLogin() + "/edit",
                        CLIENT_ID,
                        params.getClient().getClientId().toString()))
                .build();
    }

    @Nullable
    @Override
    protected List<IntToolFreelancerCard> getMassData() {
        List<Freelancer> freelancers = freelancerService.getFreelancers(enabledFreelancers().build());
        return converterService.getIntToolFreelancerCards(freelancers);
    }

    private Validator<String, Defect> validateLogin(Client client) {
        return login -> {
            ItemValidationBuilder<String, Defect> vb = ItemValidationBuilder.of(login);

            vb.check(notNull());
            if (vb.getResult().hasAnyErrors()) {
                return vb.getResult();
            }
            vb.check(Constraint.fromPredicate(c -> RbacRole.CLIENT.equals(client.getRole()), mustBeClient()),
                    When.isValid());

            if (vb.getResult().hasAnyErrors()) {
                return vb.getResult();
            }

            Freelancer freelancer = freelancerService.getFreelancers(Collections.singleton(client.getClientId()))
                    .stream()
                    .findFirst()
                    .orElse(null);
            //если создаём нового, текущий клиент не должен быть фрилансером
            vb.check(s -> freelancer != null ? clientIsAlreadyFreelancer() : null);
            return vb.getResult();
        };
    }
}
