package ru.yandex.direct.internaltools.tools.freelancer.tool;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.freelancer.model.FreelancerSkillOffer;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerSkillOffersService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolProcessingException;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.freelancer.model.IntToolSkillOfferChange;
import ru.yandex.direct.internaltools.tools.freelancer.model.IntToolSkillOfferView;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.internaltools.tools.freelancer.container.IntToolFreelanceSkillConverter.stringToFreelancerSkillId;
import static ru.yandex.direct.internaltools.tools.freelancer.container.IntToolFreelanceSkillConverter.toFreelancerSkillOffer;
import static ru.yandex.direct.internaltools.tools.freelancer.container.IntToolFreelanceSkillConverter.toIntToolSkillOffer;
import static ru.yandex.direct.internaltools.tools.freelancer.model.IntToolSkillOfferChange.DATE_FIELD_NAME;
import static ru.yandex.direct.internaltools.tools.freelancer.model.IntToolSkillOfferChange.DELETE_SKILL_OPERATION_VALUE;
import static ru.yandex.direct.internaltools.tools.freelancer.model.IntToolSkillOfferChange.OFFER_FIELD_NAME;
import static ru.yandex.direct.internaltools.tools.freelancer.model.IntToolSkillOfferChange.PRICE_FIELD_NAME;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Tool(
        name = "Редактирование списка услуг фрилансера",
        label = "freelancer_change_skills",
        description = "Если заполнить только поле FreelancerId, то отчёт покажет текущий набор навыков этого фрилансера.",
        consumes = IntToolSkillOfferChange.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.FREELANCER)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.SUPERREADER, InternalToolAccessRole.MANAGER,
        InternalToolAccessRole.SUPPORT})
@ParametersAreNonnullByDefault
public class FreelancerSkillOfferEditTool extends MassInternalTool<IntToolSkillOfferChange, IntToolSkillOfferView> {
    // все услуги фрилансеров, независимо от страны, должны быть оценены в рублях, подробнее здесь: DIRECT-83462
    private static final CurrencyCode CURRENCY_CODE = CurrencyCode.RUB;

    private final FreelancerSkillOffersService freelancerSkillOffersService;

    @Autowired
    public FreelancerSkillOfferEditTool(
            FreelancerSkillOffersService freelancerSkillOffersService) {
        this.freelancerSkillOffersService = freelancerSkillOffersService;
    }

    @Override
    public ValidationResult<IntToolSkillOfferChange, Defect> validate(
            IntToolSkillOfferChange intToolNewSkillOffer) {
        ItemValidationBuilder<IntToolSkillOfferChange, Defect> validationBuilder =
                ItemValidationBuilder.of(intToolNewSkillOffer);
        if (intToolNewSkillOffer.getFreelancerId() == null) {
            return validationBuilder.getResult();
        }
        if (isDeleteOperation(intToolNewSkillOffer)) {
            validationBuilder.check(notNull());
            validationBuilder.item(intToolNewSkillOffer.getStrSkillId(), OFFER_FIELD_NAME)
                    .check(notNull())
                    .check(notBlank());
            return validationBuilder.getResult();
        }
        // если ни одно поле, кроме FreelancerId, не заполнено, значит пользователь хотел просто получить список услуг этого фрилансера
        if (intToolNewSkillOffer.getStrSkillId() == null
                && intToolNewSkillOffer.getDuration() == null
                && intToolNewSkillOffer.getPrice() == null) {
            return validationBuilder.getResult();
        }
        //а вот если заполнено хоть одно, то должны быть заполнены и остальные
        validationBuilder.check(notNull());
        validationBuilder.item(intToolNewSkillOffer.getStrSkillId(), OFFER_FIELD_NAME)
                .check(notNull())
                .check(notBlank());
        validationBuilder.item(intToolNewSkillOffer.getDuration(), DATE_FIELD_NAME)
                .check(notNull());
        validationBuilder.item(intToolNewSkillOffer.getPrice(), PRICE_FIELD_NAME)
                .check(notNull())
                .check(notLessThan(0L));
        return validationBuilder.getResult();
    }

    @Override
    protected List<IntToolSkillOfferView> getMassData(IntToolSkillOfferChange parameter) {
        if (parameter.getFreelancerId() == null) {
            return super.getMassData();
        }
        Long freelancerId = parameter.getFreelancerId();
        String strSkillId = parameter.getStrSkillId();
        if (strSkillId != null) {
            if (isDeleteOperation(parameter)) {
                deleteSkillOffer(parameter);
            } else {
                addOrUpdateSkillOffer(parameter);
            }
        }
        List<FreelancerSkillOffer> freelancerSkillsOffers =
                freelancerSkillOffersService.getFreelancerSkillsOffers(singletonList(freelancerId));
        return mapList(freelancerSkillsOffers,
                freelancerSkillOffer -> toIntToolSkillOffer(freelancerSkillOffer, CURRENCY_CODE));
    }

    private void addOrUpdateSkillOffer(IntToolSkillOfferChange parameter) {
        ClientId clientId = ClientId.fromLong(parameter.getFreelancerId());
        FreelancerSkillOffer freelancerSkillOffer = toFreelancerSkillOffer(parameter);
        MassResult<FreelancerSkillOffer> result =
                freelancerSkillOffersService.setFreelancerSkillOffer(clientId, singletonList(freelancerSkillOffer));
        if (hasValidationIssues(result)) {
            throw new InternalToolValidationException("").withValidationResult(result.getValidationResult());
        }
    }

    private void deleteSkillOffer(IntToolSkillOfferChange parameter) {
        long skillId = stringToFreelancerSkillId(parameter.getStrSkillId());
        ClientId clientId = ClientId.fromLong(parameter.getFreelancerId());
        int result = freelancerSkillOffersService.deleteFreelancerSkillsOffer(clientId, singletonList(skillId));
        if (result == 0) {
            throw new InternalToolProcessingException(
                    String.format("Услуга со SkillId=%2$d у фрилансера с FreelancerId=%1$d не найдена.",
                            parameter.getFreelancerId(), skillId));
        }
    }

    private boolean isDeleteOperation(IntToolSkillOfferChange parameter) {
        return parameter.getOperation().equals(DELETE_SKILL_OPERATION_VALUE);
    }
}
