package ru.yandex.direct.internaltools.tools.freelancer.tool;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BinaryOperator;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.freelancer.model.Freelancer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerBase;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCard;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCardModeration;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificate;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificateType;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerContacts;
import ru.yandex.direct.core.entity.freelancer.operation.FreelancerCardAddOperation;
import ru.yandex.direct.core.entity.freelancer.operation.FreelancerUpdateOperation;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerCardService;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerClientAvatarService;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerService;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerUpdateService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.freelancer.model.FreelancerUpdateParameters;
import ru.yandex.direct.internaltools.tools.freelancer.model.IntToolFreelancerCard;
import ru.yandex.direct.internaltools.tools.freelancer.service.IntToolFreelancerConverterService;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.Comparator.comparing;
import static org.apache.commons.lang.StringUtils.trimToNull;
import static ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificateType.DIRECT;
import static ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificateType.METRIKA;
import static ru.yandex.direct.core.entity.freelancer.model.FreelancersCardStatusModerate.ACCEPTED;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.core.validation.constraints.Constraints.validLogin;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.validation.Predicates.isDouble;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Tool(
        name = "2 Изменить параметры фрилансера",
        label = "update_freelancer",
        description = "Изменение параметров фрилансера. Если указать только ClientID, "
                + "то отчёт покажет текущее состояние указанного фрилансера.",
        consumes = FreelancerUpdateParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.FREELANCER)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.MANAGER,
        InternalToolAccessRole.SUPPORT})
@ParametersAreNonnullByDefault
public class FreelancerUpdateTool extends MassInternalTool<FreelancerUpdateParameters, IntToolFreelancerCard> {
    private static final int NO_CERT = 0;

    private final FreelancerUpdateService freelancerUpdateService;
    private final FreelancerCardService freelancerCardService;
    private final FreelancerService freelancerService;
    private final FreelancerClientAvatarService freelancerClientAvatarService;
    private final IntToolFreelancerConverterService converterService;

    public FreelancerUpdateTool(FreelancerUpdateService freelancerUpdateService,
                                FreelancerCardService freelancerCardService,
                                FreelancerService freelancerService,
                                FreelancerClientAvatarService freelancerClientAvatarService,
                                IntToolFreelancerConverterService converterService) {
        this.freelancerUpdateService = freelancerUpdateService;
        this.freelancerCardService = freelancerCardService;
        this.freelancerService = freelancerService;
        this.freelancerClientAvatarService = freelancerClientAvatarService;
        this.converterService = converterService;
    }

    @Override
    public ValidationResult<FreelancerUpdateParameters, Defect> validate(FreelancerUpdateParameters params) {
        ItemValidationBuilder<FreelancerUpdateParameters, Defect> vb = ItemValidationBuilder.of(params, Defect.class);
        // Единственный параметр, который должен присутствовать -- clientId
        vb.item(params.getClientId(), "clientId")
                .check(notNull())
                .check(validId());

        vb.item(params.getCertLogin(), "certLogin")
                .check(validLogin(), When.notNull());

        ItemValidationBuilder<String, Defect> ratingVb = vb.item(params.getRating(), "rating");
        ratingVb.check(Constraint.fromPredicate(isDouble(), CommonDefects.invalidValue()));
        if (params.getRating() != null && !ratingVb.getResult().hasAnyErrors()) {
            ratingVb.item(new BigDecimal(params.getRating()), "num")
                    .check(inRange(BigDecimal.ZERO, BigDecimal.valueOf(5)));
        }
        vb.item(params.getFirstName(), "firstName")
                .check(notBlank());
        vb.item(params.getSecondName(), "secondName")
                .check(notBlank());
        vb.item(params.getBrief(), "brief")
                .check(notBlank())
                .check(notNull());
        vb.item(params.getTown(), "town")
                .check(notBlank());
        vb.item(params.getEmail(), "email")
                .check(notBlank());
        vb.item(params.getPhone(), "phone")
                .check(notBlank());
        return vb.getResult();
    }

    @Override
    protected List<IntToolFreelancerCard> getMassData(FreelancerUpdateParameters p) {
        Long freelancerId = p.getClientId();
        ClientId clientId = ClientId.fromLong(freelancerId);
        Freelancer existingFreelancer = freelancerService.getFreelancerWithNewestCard(clientId.asLong());
        checkNotNull(existingFreelancer, "Can't find freelancer by id %s", freelancerId);
        FreelancerCard existingCard = existingFreelancer.getCard();
        // Если не указаны параметры для модификации, то просто вернём текущее состояние фрилансера
        FreelancerCard cardChanges = new FreelancerCard()
                .withFreelancerId(clientId.asLong())
                .withBriefInfo(p.getBrief());
        if (hasNonNullContacts(p)) {
            // обновим переданные параметры контактов у существующего объекта, чтобы не затереть при сохранении
            FreelancerContacts contactsChanges =
                    Optional.ofNullable(existingCard).map(FreelancerCard::getContacts).orElse(new FreelancerContacts());
            if (p.getTown() != null) {
                contactsChanges.setTown(p.getTown());
            }
            if (p.getPhone() != null) {
                contactsChanges.setPhone(p.getPhone());
            }
            if (p.getEmail() != null) {
                contactsChanges.setEmail(p.getEmail());
            }
            // эти контакты очищаем, если передали пробельную строчку
            if (p.getIcq() != null) {
                contactsChanges.setIcq(trimToNull(p.getIcq()));
            }
            if (p.getWhatsApp() != null) {
                contactsChanges.setWhatsApp(trimToNull(p.getWhatsApp()));
            }
            if (p.getTelegram() != null) {
                contactsChanges.setTelegram(trimToNull(p.getTelegram()));
            }
            if (p.getSiteUrl() != null) {
                contactsChanges.setSiteUrl(trimToNull(p.getSiteUrl()));
            }
            if (p.getSkype() != null) {
                contactsChanges.setSkype(trimToNull(p.getSkype()));
            }
            if (p.getViber() != null) {
                contactsChanges.setViber(trimToNull(p.getViber()));
            }
            cardChanges.withContacts(contactsChanges);
        }

        FreelancerBase freelancerChanges = new FreelancerBase()
                .withId(freelancerId)
                .withFirstName(p.getFirstName())
                .withSecondName(p.getSecondName())
                .withRegionId(p.getRegionId())
                .withRating(p.getRating() == null ? null : Double.valueOf(p.getRating()))
                .withIsSearchable(p.getVisibility() == null ? null : p.getVisibility().boolValue())
                .withCertLogin(p.getCertLogin());

        // если задано, сертификаты как и контакты обновляем вместе
        Long directCertId = p.getDirectCertId();
        Long metrikaCertId = p.getMetrikaCertId();
        if (directCertId != null || metrikaCertId != null) {
            // в сертификатах могут по ошибке появляться дубли (DIRECT-116833)
            // из однотипных сертификатов выбираем тот, кто новее (больше ID)
            var certMergingFunction = BinaryOperator.maxBy(comparing(FreelancerCertificate::getCertId));
            Map<FreelancerCertificateType, FreelancerCertificate> newCertificatesByType =
                    StreamEx.of(nvl(existingFreelancer.getCertificates(), emptyList()))
                            .mapToEntry(FreelancerCertificate::getType, Function.identity())
                            .toMap(certMergingFunction);
            if (directCertId != null) {
                if (directCertId == NO_CERT) {
                    newCertificatesByType.remove(DIRECT);
                } else {
                    newCertificatesByType.put(DIRECT,
                            new FreelancerCertificate()
                                    .withCertId(directCertId)
                                    .withType(DIRECT));
                }
            }
            if (metrikaCertId != null) {
                if (metrikaCertId == NO_CERT) {
                    newCertificatesByType.remove(METRIKA);
                } else {
                    newCertificatesByType.put(METRIKA,
                            new FreelancerCertificate()
                                    .withCertId(metrikaCertId)
                                    .withType(METRIKA));
                }
            }
            freelancerChanges.withCertificates(new ArrayList<>(newCertificatesByType.values()));
        }
        FreelancerUpdateOperation freelancerUpdateOperation =
                freelancerUpdateService.getFreelancerUpdateOperation(clientId, freelancerChanges);
        FreelancerCardAddOperation freelancerCardAddOperation = getCardAddOperation(clientId, cardChanges);

        List<Operation<Long>> operations = StreamEx.of(freelancerUpdateOperation, freelancerCardAddOperation)
                .nonNull()
                .map(it -> (Operation<Long>) it)
                .toList();

        ValidationResult<?, Defect> validationResult = prepareAndApply(operations);
        if (validationResult != null) {
            throw new InternalToolValidationException("")
                    .withValidationResult(validationResult);
        }
        if (p.getAvatarImage() != null && p.getAvatarImage().length > 0) {
            // todo maxlog: обновление аватарки хорошо бы тоже в операцию обернуть
            Result<Long> updateAvatarResult =
                    freelancerClientAvatarService.updateAvatar(clientId, p.getAvatarImage());
            if (hasValidationIssues(updateAvatarResult)) {
                throw new InternalToolValidationException("Error on updating client avatar")
                        .withValidationResult(updateAvatarResult.getValidationResult());
            }
        }
        if (freelancerCardAddOperation != null) {
            // Ставим карточке статус ACCEPTED.
            moderateChangedCard(freelancerId, freelancerCardAddOperation);
        }
        // получим актуальное состояние фрилансера, так как могли что-то поменять
        List<Freelancer> actualFreelancers = freelancerService.getFreelancers(singleton(clientId.asLong()));
        return getIntToolFreelancerCards(actualFreelancers);
    }

    /**
     * @return {@code null}, если не было изменений. Иначе -- подготовленный экземпляр
     * {@link FreelancerCardAddOperation}
     */
    @Nullable
    private FreelancerCardAddOperation getCardAddOperation(ClientId clientId, FreelancerCard cardChanges) {
        boolean hasCardModification =
                cardChanges.getContacts() != null ||
                        cardChanges.getBriefInfo() != null;
        if (!hasCardModification) {
            return null;
        }
        return freelancerCardService.getChangedCardAddOperation(clientId, cardChanges);
    }

    private void moderateChangedCard(Long freelancerId, FreelancerCardAddOperation freelancerCardAddOperation) {
        Optional<MassResult<Long>> result = freelancerCardAddOperation.getResult();
        if (!result.isPresent()) {
            throw new InternalToolValidationException("The card couldn't be added.");
        }
        Long addedCardId = result.get().get(0).getResult();
        FreelancerCardModeration moderationResult = new FreelancerCard()
                .withId(addedCardId)
                .withFreelancerId(freelancerId)
                .withStatusModerate(ACCEPTED)
                .withDeclineReason(emptySet());
        freelancerCardService.applyModerationResult(moderationResult);
    }

    @Nullable
    private ValidationResult<?, Defect> prepareAndApply(List<Operation<Long>> operations) {
        boolean success = true;
        for (Operation<Long> operation : operations) {
            Optional<MassResult<Long>> prepareResult = operation.prepare();
            if (prepareResult.isPresent()) {
                success = false;
                break;
            }
        }
        if (!success) {
            return mergedValidationResult(operations);
        }
        operations.forEach(Operation::apply);
        return null;
    }

    private ValidationResult<?, Defect> mergedValidationResult(List<Operation<Long>> operations) {
        ValidationResult<?, Defect> result = null;

        for (Operation<Long> operation : operations) {
            Optional<MassResult<Long>> prepareResult = operation.getResult();
            if (!prepareResult.isPresent()) {
                continue;
            }
            ValidationResult<?, Defect> validationResult = prepareResult.get().getValidationResult();
            if (result == null) {
                result = new ValidationResult<>(validationResult);
            } else {
                result.merge(validationResult);
            }
        }
        boolean hasErrors = result != null && result.hasAnyErrors();
        checkState(hasErrors, "No errors found in operations.");
        return result;
    }

    private List<IntToolFreelancerCard> getIntToolFreelancerCards(List<Freelancer> freelancers) {
        return converterService.getIntToolFreelancerCards(freelancers);
    }

    private boolean hasNonNullContacts(FreelancerUpdateParameters p) {
        return p.getTown() != null || p.getPhone() != null || p.getEmail() != null ||
                p.getIcq() != null || p.getWhatsApp() != null || p.getTelegram() != null ||
                p.getSiteUrl() != null || p.getSkype() != null || p.getViber() != null;
    }
}
