package ru.yandex.direct.internaltools.tools.hourglass;

import java.time.LocalDateTime;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.client.HourglassClient;
import ru.yandex.direct.hourglass.client.SchedulerInstance;
import ru.yandex.direct.hourglass.storage.JobStatus;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.hourglass.container.TriggerInfo;
import ru.yandex.direct.internaltools.tools.hourglass.container.TriggersListParameters;

import static java.util.stream.Collectors.toMap;

@Tool(
        name = "Список всех задач в планировщике",
        label = "hourglass_list_all_triggers",
        description = "Отображает все задачи шедулера\n" +
                "Readme по ydb и примеры запросов - https://a.yandex-team.ru/arc/trunk/arcadia/direct/jobs/README.md",
        consumes = TriggersListParameters.class,
        type = InternalToolType.REPORT
)
@Action(InternalToolAction.SHOW)
@Category(InternalToolCategory.HOURGLASS)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.SUPERREADER, InternalToolAccessRole.DEVELOPER})
public class HourglassListTriggersTool extends MassInternalTool<TriggersListParameters, TriggerInfo> {
    private static final Comparator<TriggerInfo> COMPARATOR =
            Comparator.comparing(TriggerInfo::getNextFireTime, Comparator.nullsLast(LocalDateTime::compareTo))
                    .thenComparing(TriggerInfo::getJobGroup, Comparator.nullsLast(String::compareTo))
                    .thenComparing(TriggerInfo::getJobDescription, Comparator.nullsLast(String::compareTo));

    private final HourglassClient hourglassClient;

    @Autowired
    public HourglassListTriggersTool(HourglassClient hourglassClient) {
        this.hourglassClient = hourglassClient;
    }

    @Override
    protected List<TriggerInfo> getMassData(@Nonnull TriggersListParameters parameter) {
        var tasks = hourglassClient.getTasks();
        Map<InstanceId, SchedulerInstance> instanceIdToInstance =
                hourglassClient.getSchedulers().stream().collect(toMap(SchedulerInstance::getInstanceId,
                        instanceInfo -> instanceInfo));
        return tasks.stream()
                .filter(el -> el.jobStatus() != JobStatus.ARCHIVED)
                .filter(el -> !parameter.getOnlyLocked() || el.jobStatus() == JobStatus.LOCKED)
                .map(task -> {
                    var schedulerInstance = task.instanceId() != null ?
                            instanceIdToInstance.get(task.instanceId()) : null;
                    return new TriggerInfo(task, schedulerInstance);
                })
                .sorted(COMPARATOR)
                .collect(Collectors.toList());
    }


}
