package ru.yandex.direct.internaltools.tools.hourglass;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.hourglass.client.HourglassClient;
import ru.yandex.direct.hourglass.storage.Job;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolParameter;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.core.input.InternalToolInput;
import ru.yandex.direct.internaltools.core.input.InternalToolInputPreProcessor;
import ru.yandex.direct.internaltools.tools.hourglass.container.JobInfo;
import ru.yandex.direct.internaltools.tools.hourglass.container.PauseJobParameter;
import ru.yandex.direct.internaltools.tools.hourglass.container.PauseOption;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.internaltools.tools.hourglass.container.RunJobParameter.JOB_ID_NAME;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;

@Tool(
        name = "Пауза и снятие с паузы джоб",
        label = "hourglass_pause_job_tool",
        description = "Позволяет поставить задачу на паузу и снять с паузы.\n" +
                "При запуске без указания \"" + JOB_ID_NAME + "\" - выведет возможные параметры запуска.\n" +
                "Чтобы поставить на паузу/снятьс паузы задачу - нужно указать один из перечисленных id.\n" +
                "Readme по ydb и примеры запросов - https://a.yandex-team.ru/arc/trunk/arcadia/direct/jobs/README.md",
        consumes = PauseJobParameter.class,
        type = InternalToolType.REPORT
)
@Action(InternalToolAction.RUN)
@Category(InternalToolCategory.HOURGLASS)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class HourglassPauseJobTool extends MassInternalTool<PauseJobParameter, JobInfo> {

    private final HourglassClient hourglassClient;

    @Autowired
    public HourglassPauseJobTool(HourglassClient hourglassClient) {
        this.hourglassClient = hourglassClient;
    }

    @Override
    public ValidationResult<PauseJobParameter, Defect> validate(PauseJobParameter parameter) {
        ItemValidationBuilder<PauseJobParameter, Defect> vb = ItemValidationBuilder.of(parameter);

        Set<String> schedulesSet =
                hourglassClient.getTasks().stream().map(e -> e.primaryId().toString())
                        .collect(Collectors.toSet());

        Predicate<PauseJobParameter> checkJobExists = p -> p.getJobParametersId().equalsIgnoreCase("all") ||
                schedulesSet.contains(p.getJobParametersId());

        return vb.item(parameter, JOB_ID_NAME)
                .check(Constraint.fromPredicate(checkJobExists, CommonDefects.invalidValue()),
                        When.isFalse(parameter.getJobParametersId().isEmpty()))
                .getResult();
    }

    @Override
    protected List<JobInfo> getMassData(PauseJobParameter p) {
        String jobName = p.getJobName();
        PauseOption pauseOption = PauseOption.fromValue(p.getPauseOptionName());

        List<Job> jobs = hourglassClient.getTasks().stream().filter(el -> el.taskId().name().equals(jobName)).collect(
                Collectors.toList());

        if (p.getJobParametersId().isEmpty()) {
            return jobs.stream().map(JobInfo::new).collect(Collectors.toList());
        } else {
            Predicate<Job> predicate;

            if (p.getJobParametersId().equalsIgnoreCase("ALL")) {
                predicate = job -> true;
            } else {
                predicate = job -> job.primaryId().toString().equals(p.getJobParametersId());
            }

            List<Job> foundKeys = jobs.stream()
                    .filter(predicate)
                    .collect(Collectors.toList());

            var pauseAction = getPauseAction(pauseOption);

            for (Job job : foundKeys) {
                pauseAction.accept(job, hourglassClient);
            }

            return foundKeys.stream().map(JobInfo::new).collect(Collectors.toList());
        }
    }

    private BiConsumer<Job, HourglassClient> getPauseAction(PauseOption pauseOption) {
        if (pauseOption == PauseOption.PAUSE) {
            return this::pauseTask;
        } else {
            return this::unPauseTask;
        }
    }

    private void pauseTask(Job job, HourglassClient hourglassClient) {
        hourglassClient.pauseTask(job);
    }

    private void unPauseTask(Job job, HourglassClient hourglassClient) {
        hourglassClient.continueTask(job);
    }

    @Component
    public class JobPauseOption implements InternalToolInputPreProcessor<String> {
        @Override
        public <T extends InternalToolParameter> InternalToolInput.Builder<T, String> preSend(
                InternalToolInput.Builder<T, String> inputBuilder) {
            List<String> optionsToChoose = new ArrayList<>();
            List<String> optionNames =
                    Arrays.stream(PauseOption.values()).map(PauseOption::value).collect(Collectors.toUnmodifiableList());
            optionsToChoose.add(null);
            optionsToChoose.addAll(optionNames);
            return inputBuilder
                    .withDefaultValue(null)
                    .withAllowedValues(optionsToChoose)
                    .addValidator(inSet(ImmutableSet.<String>builder().addAll(optionNames).build()));
        }
    }
}
