package ru.yandex.direct.internaltools.tools.hourglass;

import java.time.Instant;
import java.time.OffsetDateTime;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.hourglass.client.HourglassClient;
import ru.yandex.direct.hourglass.storage.Job;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.hourglass.container.JobInfo;
import ru.yandex.direct.internaltools.tools.hourglass.container.RunJobParameter;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.internaltools.tools.hourglass.container.RunJobParameter.JOB_ID_NAME;

@Tool(
        name = "Запуск джоб по кнопке",
        label = "hourglass_run_job_tool",
        description = "Позволяет запустить задачу.\n" +
                "При запуске без указания \"" + JOB_ID_NAME + "\" - выведет возможные параметры запуска.\n" +
                "Чтобы запустить задачу - нужно указать один из перечисленных id.\n" +
                "Readme по ydb и примеры запросов - https://a.yandex-team.ru/arc/trunk/arcadia/direct/jobs/README.md",
        consumes = RunJobParameter.class,
        type = InternalToolType.REPORT
)
@Action(InternalToolAction.RUN)
@Category(InternalToolCategory.HOURGLASS)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class HourglassRunJobTool extends MassInternalTool<RunJobParameter, JobInfo> {

    private final HourglassClient hourglassClient;

    @Autowired
    public HourglassRunJobTool(HourglassClient hourglassClient) {
        this.hourglassClient = hourglassClient;
    }

    @Override
    public ValidationResult<RunJobParameter, Defect> validate(RunJobParameter parameter) {
        ItemValidationBuilder<RunJobParameter, Defect> vb = ItemValidationBuilder.of(parameter);

        Set<String> schedulesSet =
                hourglassClient.getTasks().stream().map(e -> e.primaryId().toString())
                        .collect(Collectors.toSet());

        Predicate<RunJobParameter> checkJobExists = p -> p.getJobParametersId().equalsIgnoreCase("all") ||
                schedulesSet.contains(p.getJobParametersId());

        return vb.item(parameter, JOB_ID_NAME)
                .check(Constraint.fromPredicate(checkJobExists, CommonDefects.invalidValue()),
                        When.isFalse(parameter.getJobParametersId().isEmpty()))
                .getResult();
    }

    @Override
    protected List<JobInfo> getMassData(RunJobParameter p) {
        String jobName = p.getJobName();
        List<Job> jobs = hourglassClient.getTasks().stream().filter(el -> el.taskId().name().equals(jobName)).collect(
                Collectors.toList());

        if (p.getJobParametersId().isEmpty()) {
            return jobs.stream().map(JobInfo::new).collect(Collectors.toList());
        } else {
            Predicate<Job> predicate;

            if (p.getJobParametersId().equalsIgnoreCase("ALL")) {
                predicate = job -> true;
            } else {
                predicate = job -> job.primaryId().toString().equals(p.getJobParametersId());
            }

            List<Job> foundKeys = jobs.stream().filter(predicate).collect(Collectors.toList());

            for (Job job : foundKeys) {
                hourglassClient.changeNextRunTime(job, Instant.now(), OffsetDateTime.now().getOffset());
            }

            return foundKeys.stream().map(JobInfo::new).collect(Collectors.toList());
        }
    }
}
