package ru.yandex.direct.internaltools.tools.hourglass.container;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;

import ru.yandex.direct.hourglass.client.SchedulerInstance;
import ru.yandex.direct.hourglass.storage.Job;

import static ru.yandex.direct.internaltools.tools.hourglass.container.TriggerInfo.INSTANCE_META_FIELD;
import static ru.yandex.direct.internaltools.tools.hourglass.container.TriggerInfo.INSTANCE_VERSION;
import static ru.yandex.direct.internaltools.tools.hourglass.container.TriggerInfo.JOB_DESCRIPTION_FIELD;
import static ru.yandex.direct.internaltools.tools.hourglass.container.TriggerInfo.JOB_GROUP_FIELD;
import static ru.yandex.direct.internaltools.tools.hourglass.container.TriggerInfo.META_FIELD;
import static ru.yandex.direct.internaltools.tools.hourglass.container.TriggerInfo.NEXT_TIME_FIELD;
import static ru.yandex.direct.internaltools.tools.hourglass.container.TriggerInfo.PREV_TIME_FIELD;
import static ru.yandex.direct.internaltools.tools.hourglass.container.TriggerInfo.SCHEDULER_NAME_FIELD;
import static ru.yandex.direct.internaltools.tools.hourglass.container.TriggerInfo.STATE_FIELD;

@JsonPropertyOrder({NEXT_TIME_FIELD, PREV_TIME_FIELD, STATE_FIELD, JOB_GROUP_FIELD, JOB_DESCRIPTION_FIELD,
        SCHEDULER_NAME_FIELD, META_FIELD, INSTANCE_META_FIELD, INSTANCE_VERSION})
public class TriggerInfo {
    static final String NEXT_TIME_FIELD = "Следующий запуск";
    static final String PREV_TIME_FIELD = "Предыдущий запуск";

    static final String STATE_FIELD = "Состояние";
    static final String JOB_DESCRIPTION_FIELD = "Параметры задачи";
    static final String JOB_GROUP_FIELD = "Группа задач";
    static final String META_FIELD = "Дополнительная информация";
    static final String INSTANCE_META_FIELD = "Информация об инстансе";
    static final String INSTANCE_VERSION = "Версия инстанса";
    static final String SCHEDULER_NAME_FIELD = "Название инстанса";

    private final String state;
    private final String jobGroup;
    private final String jobDescription;
    private final LocalDateTime nextFireTime;
    private final LocalDateTime prevFireTime;
    private final String schedulerName;
    private final String meta;
    private final String instanceMeta;
    private final String instanceVersion;

    public TriggerInfo(Job job, SchedulerInstance schedulerInstance) {
        this(
                job.instanceId().toString(),
                job.taskId().name(),
                job.taskId().param(),
                job.jobStatus().toString(),
                job.nextRun(),
                job.taskProcessingResult() != null ? job.taskProcessingResult().lastStartTime() : null,
                job.meta(),
                schedulerInstance != null ? schedulerInstance.getInstanceMeta() : null,
                schedulerInstance != null ? schedulerInstance.getVersion() : null
        );
    }

    public TriggerInfo(String schedulerName, String jobGroup, String jobDescription, String state,
                       Instant nextFireTime, Instant prevFireTime, String meta, String instanceMeta,
                       String instanceVersion) {
        this.schedulerName = schedulerName;
        this.jobGroup = jobGroup;
        this.jobDescription = jobDescription;
        this.state = state;
        this.nextFireTime = getLocalDateTimeFromInstant(nextFireTime);
        this.prevFireTime = getLocalDateTimeFromInstant(prevFireTime);
        this.meta = meta;
        this.instanceMeta = instanceMeta;
        this.instanceVersion = instanceVersion;
    }


    @JsonProperty(NEXT_TIME_FIELD)
    public LocalDateTime getNextFireTime() {
        return nextFireTime;
    }

    @JsonProperty(PREV_TIME_FIELD)
    public LocalDateTime getPrevFireTime() {
        return prevFireTime;
    }

    @JsonProperty(STATE_FIELD)
    public String getState() {
        return state;
    }

    @JsonProperty(JOB_GROUP_FIELD)
    public String getJobGroup() {
        return jobGroup;
    }

    @JsonProperty(JOB_DESCRIPTION_FIELD)
    public String getJobDescription() {
        return jobDescription;
    }

    @JsonProperty(SCHEDULER_NAME_FIELD)
    public String getSchedulerName() {
        return schedulerName;
    }

    @JsonProperty(META_FIELD)
    public String getMeta() {
        return meta;
    }

    @JsonProperty(INSTANCE_META_FIELD)
    public String getInstanceMeta() {
        return instanceMeta;
    }

    @JsonProperty(INSTANCE_VERSION)
    public String getInstanceVersion() {
        return instanceVersion;
    }

    private LocalDateTime getLocalDateTimeFromInstant(Instant instant) {
        if (instant == null) {
            return null;
        }
        return LocalDateTime.ofInstant(instant, ZoneId.systemDefault());
    }
}
